<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\GameMatch;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Tournament;
use App\Models\Player;
use App\Models\Team;
use App\Models\User;

class MatchAdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:matches.view')->only(['index', 'show']);
        $this->middleware('permission:matches.create')->only(['create', 'store']);
        $this->middleware('permission:matches.edit')->only(['edit', 'update', 'updateScore', 'updateMultiple']);
        $this->middleware('permission:matches.delete')->only(['destroy', 'archiveMatch', 'unarchiveMatch']);
        $this->middleware('permission:matches.start')->only(['startMatch', 'pauseMatch', 'resumeMatch']);
        $this->middleware('permission:matches.complete')->only(['completeMatch']);
        $this->middleware('permission:matches.verify')->only(['verifyMatch']);
    }

    public function index(Request $request)
    {
        $query = GameMatch::with(['tournament', 'bracket', 'player1', 'player2', 'team1', 'team2', 'winner', 'winningTeam', 'referee']);

        // Advanced filtering
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('match_code', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        if ($request->filled('progress_status') && $request->progress_status !== 'all') {
            $query->where('progress_status', $request->progress_status);
        }

        if ($request->filled('match_type') && $request->match_type !== 'all') {
            $query->where('match_type', $request->match_type);
        }

        if ($request->filled('format') && $request->format !== 'all') {
            $query->where('format', $request->format);
        }

        if ($request->filled('tournament_id') && $request->tournament_id !== 'all') {
            $query->where('tournament_id', $request->tournament_id);
        }

        if ($request->filled('bracket_id') && $request->bracket_id !== 'all') {
            $query->where('bracket_id', $request->bracket_id);
        }

        if ($request->filled('player_id') && $request->player_id !== 'all') {
            $query->where(function($q) use ($request) {
                $q->where('player1_id', $request->player_id)
                  ->orWhere('player2_id', $request->player_id);
            });
        }

        if ($request->filled('team_id') && $request->team_id !== 'all') {
            $query->where(function($q) use ($request) {
                $q->where('team1_id', $request->team_id)
                  ->orWhere('team2_id', $request->team_id);
            });
        }

        if ($request->filled('is_streamed') && $request->is_streamed !== 'all') {
            $query->where('is_streamed', $request->is_streamed === 'true');
        }

        if ($request->filled('verification_status') && $request->verification_status !== 'all') {
            $query->where('verification_status', $request->verification_status);
        }

        if ($request->filled('date_from')) {
            $query->where('scheduled_time', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('scheduled_time', '<=', $request->date_to);
        }

        if ($request->filled('referee_id') && $request->referee_id !== 'all') {
            $query->where('referee_id', $request->referee_id);
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'scheduled_time');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $matches = $query->paginate($request->get('per_page', 15))->withQueryString();

        // Get related data for filters
        $tournaments = Tournament::select('id', 'name')->get();
        $players = Player::select('id', 'display_name', 'tournament_id')->get();
        $teams = Team::select('id', 'name', 'tournament_id')->get();
        $referees = User::whereHas('roles', function($q) {
            $q->where('name', 'referee');
        })->select('id', 'name')->get();

        return Inertia::render('admin/matches/index', [
            'matches' => $matches,
            'filters' => $request->only([
                'search', 'status', 'progress_status', 'match_type', 'format', 
                'tournament_id', 'bracket_id', 'player_id', 'team_id', 
                'is_streamed', 'verification_status', 'date_from', 'date_to', 
                'referee_id', 'sort_by', 'sort_order'
            ]),
            'tournaments' => $tournaments,
            'players' => $players,
            'teams' => $teams,
            'referees' => $referees,
        ]);
    }

    public function create()
    {
        $tournaments = Tournament::select('id', 'name')->get();
        $players = Player::select('id', 'display_name', 'tournament_id')->get();
        $teams = Team::select('id', 'name', 'tournament_id')->get();
        $referees = User::whereHas('roles', function($q) {
            $q->where('name', 'referee');
        })->select('id', 'name')->get();

        return Inertia::render('admin/matches/create', [
            'tournaments' => $tournaments,
            'players' => $players,
            'teams' => $teams,
            'referees' => $referees,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'match_code' => 'nullable|string|max:255|unique:matches,match_code',
            'match_type' => 'required|in:individual,team,mixed,group_stage,knockout_stage,semi_final,final,ladies_final,kids_special,ladies_special,entertainment,consolation,mens_final,wazee_final',
            'format' => 'required|in:best_of_1,best_of_3,best_of_5,best_of_7,first_to_win,time_limit',
            'best_of_games' => 'required|integer|min:1',
            'time_limit_minutes' => 'nullable|integer|min:1',
            'rules' => 'nullable|array',
            'settings' => 'nullable|array',
            'description' => 'nullable|string',
            'notes' => 'nullable|string',
            'tournament_id' => 'required|exists:tournaments,id',
            'bracket_id' => 'nullable|exists:brackets,id',
            'round' => 'required|integer|min:1',
            'match_number' => 'required|integer|min:1',
            'player1_id' => 'nullable|exists:players,id',
            'player2_id' => 'nullable|exists:players,id',
            'team1_id' => 'nullable|exists:teams,id',
            'team2_id' => 'nullable|exists:teams,id',
            'scheduled_time' => 'required|date',
            'estimated_duration_minutes' => 'nullable|integer|min:1',
            'is_streamed' => 'boolean',
            'stream_url' => 'nullable|url',
            'twitch_url' => 'nullable|url',
            'youtube_url' => 'nullable|url',
            'allow_spectators' => 'boolean',
            'referee_id' => 'nullable|exists:users,id',
            'created_by' => 'nullable|exists:users,id',
        ]);

        // Set default values
        $validated['status'] = 'scheduled';
        $validated['progress_status'] = 'not_started';
        $validated['verification_status'] = 'pending';
        $validated['created_by'] = auth()->id();

        // Generate match code if not provided
        if (empty($validated['match_code'])) {
            $validated['match_code'] = 'MATCH-' . strtoupper(\Str::random(8));
        }

        GameMatch::create($validated);

        return redirect()->route('admin.matches.index')->with('success', 'Match created successfully!');
    }

    public function show(GameMatch $match)
    {
        $match->load([
            'tournament', 'bracket', 'player1', 'player2', 'team1', 'team2', 
            'winner', 'winningTeam', 'referee', 'verifiedBy', 'createdBy', 'updatedBy'
        ]);

        return Inertia::render('admin/matches/show', [
            'match' => $match,
            'detailedStats' => $match->getDetailedStats(),
        ]);
    }

    public function edit(GameMatch $match)
    {
        $match->load(['tournament', 'bracket', 'player1', 'player2', 'team1', 'team2']);

        $tournaments = Tournament::select('id', 'name')->get();
        $players = Player::select('id', 'display_name', 'tournament_id')->get();
        $teams = Team::select('id', 'name', 'tournament_id')->get();
        $referees = User::whereHas('roles', function($q) {
            $q->where('name', 'referee');
        })->select('id', 'name')->get();

        return Inertia::render('admin/matches/edit', [
            'match' => $match,
            'tournaments' => $tournaments,
            'players' => $players,
            'teams' => $teams,
            'referees' => $referees,
        ]);
    }

    public function update(Request $request, GameMatch $match)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',    
            'match_code' => 'nullable|string|max:255|unique:matches,match_code,' . $match->id,
            'match_type' => 'required|in:individual,team,mixed,group_stage,knockout_stage,semi_final,final,ladies_final,kids_special,ladies_special,entertainment,consolation,mens_final,wazee_final',
            'format' => 'required|in:best_of_1,best_of_3,best_of_5,best_of_7,first_to_win,time_limit',
            'best_of_games' => 'required|integer|min:1',
            'time_limit_minutes' => 'nullable|integer|min:1',
            'rules' => 'nullable|array',
            'settings' => 'nullable|array',
            'description' => 'nullable|string',
            'notes' => 'nullable|string',
            'tournament_id' => 'required|exists:tournaments,id',
            'bracket_id' => 'nullable|exists:brackets,id',
            'round' => 'required|integer|min:1',
            'match_number' => 'required|integer|min:1',
            'player1_id' => 'nullable|exists:players,id',
            'player2_id' => 'nullable|exists:players,id',
            'team1_id' => 'nullable|exists:teams,id',
            'team2_id' => 'nullable|exists:teams,id',
            'player1_score' => 'nullable|integer|min:0',
            'player2_score' => 'nullable|integer|min:0',
            'game_scores' => 'nullable|array',
            'round_scores' => 'nullable|array',
            'scheduled_time' => 'required|date',
            'estimated_duration_minutes' => 'nullable|integer|min:1',
            'is_streamed' => 'boolean',
            'stream_url' => 'nullable|url',
            'twitch_url' => 'nullable|url',
            'youtube_url' => 'nullable|url',
            'allow_spectators' => 'boolean',
            'referee_id' => 'nullable|exists:users,id',
            'updated_by' => 'nullable|exists:users,id',
        ]);

        $validated['updated_by'] = auth()->id();
        $validated['last_activity_at'] = now();

        $match->update($validated);

        return redirect()->route('admin.matches.index')->with('success', 'Match updated successfully!');
    }

    public function destroy(GameMatch $match)
    {
        $match->delete();
        return redirect()->route('admin.matches.index')->with('success', 'Match deleted successfully!');
    }

    // Advanced match management methods
    public function startMatch(GameMatch $match)
    {
        try {
            $match->startMatch();
            return redirect()->back()->with('success', 'Match started successfully!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function pauseMatch(GameMatch $match)
    {
        try {
            $match->pauseMatch();
            return redirect()->back()->with('success', 'Match paused successfully!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function resumeMatch(GameMatch $match)
    {
        try {
            $match->resumeMatch();
            return redirect()->back()->with('success', 'Match resumed successfully!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function completeMatch(Request $request, GameMatch $match)
    {
        $validated = $request->validate([
            'winner_id' => 'nullable|exists:players,id',
            'winning_team_id' => 'nullable|exists:teams,id',
            'result_type' => 'required|in:normal,forfeit,disqualification,technical_win,draw,no_contest',
            'result_notes' => 'nullable|string',
        ]);

        try {
            $match->completeMatch(
                $validated['winner_id'],
                $validated['winning_team_id'],
                $validated['result_type']
            );

            if (!empty($validated['result_notes'])) {
                $match->update(['result_notes' => $validated['result_notes']]);
            }

            return redirect()->back()->with('success', 'Match completed successfully!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function updateScore(Request $request, GameMatch $match)
    {
        $validated = $request->validate([
            'player1_score' => 'nullable|integer|min:0',
            'player2_score' => 'nullable|integer|min:0',
            'game_scores' => 'nullable|array',
            'round_scores' => 'nullable|array',
        ]);

        try {
            $match->updateScore(
                $validated['player1_score'],
                $validated['player2_score'],
                $validated['game_scores'] ?? null
            );

            return redirect()->back()->with('success', 'Score updated successfully!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function verifyMatch(Request $request, GameMatch $match)
    {
        $validated = $request->validate([
            'verification_status' => 'required|in:verified,disputed,under_review',
            'admin_notes' => 'nullable|array',
        ]);

        $validated['verified_by'] = auth()->id();
        $validated['verified_at'] = now();

        $match->update($validated);

        return redirect()->back()->with('success', 'Match verification status updated!');
    }

    public function archiveMatch(GameMatch $match)
    {
        $match->archive();
        return redirect()->back()->with('success', 'Match archived successfully!');
    }

    public function unarchiveMatch(GameMatch $match)
    {
        $match->unarchive();
        return redirect()->back()->with('success', 'Match unarchived successfully!');
    }

    public function updateMultiple(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:matches,id',
            'action' => 'required|in:archive,unarchive,delete,start,pause,resume,complete',
            'data' => 'nullable|array',
        ]);

        $matches = GameMatch::whereIn('id', $validated['ids'])->get();
        $successCount = 0;
        $errorCount = 0;

        foreach ($matches as $match) {
            try {
                switch ($validated['action']) {
                    case 'archive':
                        $match->archive();
                        break;
                    case 'unarchive':
                        $match->unarchive();
                        break;
                    case 'delete':
                        $match->delete();
                        break;
                    case 'start':
                        $match->startMatch();
                        break;
                    case 'pause':
                        $match->pauseMatch();
                        break;
                    case 'resume':
                        $match->resumeMatch();
                        break;
                    case 'complete':
                        $match->completeMatch();
                        break;
                }
                $successCount++;
            } catch (\Exception $e) {
                $errorCount++;
            }
        }

        $message = "Successfully processed {$successCount} matches";
        if ($errorCount > 0) {
            $message .= ", {$errorCount} failed";
        }

        return redirect()->back()->with('success', $message);
    }
} 