<?php

namespace App\Http\Controllers\Admin\Frontend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Frontend\VlogVideo;

class VlogVideoAdminController extends Controller
{
    public function index()
    {
        $vlogVideos = VlogVideo::withTrashed()
            ->orderBy('sort_order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($video) {
                return [
                    'id' => $video->id,
                    'title' => $video->title,
                    'description' => $video->description,
                    'platform' => $video->platform,
                    'platform_icon' => $video->platform_icon,
                    'video_url' => $video->video_url,
                    'embed_url' => $video->embed_code,
                    'thumbnail_url' => $video->thumbnail_url,
                    'duration' => $video->formatted_duration,
                    'views' => $video->formatted_views,
                    'likes' => $video->likes,
                    'category' => $video->category,
                    'tags' => $video->tags,
                    'featured' => $video->featured,
                    'status' => $video->status,
                    'published_at' => $video->published_at?->format('Y-m-d'),
                    'sort_order' => $video->sort_order,
                    'created_at' => $video->created_at->format('Y-m-d H:i'),
                    'updated_at' => $video->updated_at->format('Y-m-d H:i'),
                    'deleted_at' => $video->deleted_at?->format('Y-m-d H:i'),
                ];
            });

        return Inertia::render('admin/frontend/vlog/index', [
            'vlogVideos' => $vlogVideos,
        ]);
    }

    public function create()
    {
        $categories = VlogVideo::getCategories();
        $platforms = VlogVideo::getPlatforms();

        return Inertia::render('admin/frontend/vlog/create', [
            'categories' => $categories,
            'platforms' => $platforms,
        ]);
    }

    public function store(Request $request)
    {
        // $validated = $request->validate([
        //     'title' => 'required|string|max:255',
        //     'description' => 'nullable|string',
        //     'platform' => 'required|string|in:' . implode(',', array_keys(VlogVideo::getPlatforms())),
        //     'video_url' => 'required|url',
        //     'thumbnail_url' => 'nullable|url',
        //     'duration' => 'nullable|integer|min:1',
        //     'views' => 'nullable|integer|min:0',
        //     'likes' => 'nullable|integer|min:0',
        //     'category' => 'required|string|in:' . implode(',', array_keys(VlogVideo::getCategories())),
        //     'tags' => 'nullable|array',
        //     'tags.*' => 'string|max:50',
        //     'featured' => 'boolean',
        //     'status' => 'required|string|in:draft,published,archived',
        //     'published_at' => 'nullable|date',
        //     'sort_order' => 'nullable|integer|min:0',
        // ]);

        $video = VlogVideo::create($request->all());

        return redirect()->route('admin.frontend.vlog.index')
            ->with('success', 'Vlog video created successfully.');
    }

    public function edit(VlogVideo $vlogVideo)
    {
        $categories = VlogVideo::getCategories();
        $platforms = VlogVideo::getPlatforms();

        $video = [
            'id' => $vlogVideo->id,
            'title' => $vlogVideo->title,
            'description' => $vlogVideo->description,
            'platform' => $vlogVideo->platform,
            'video_url' => $vlogVideo->video_url,
            'thumbnail_url' => $vlogVideo->thumbnail_url,
            'duration' => $vlogVideo->duration,
            'views' => $vlogVideo->views,
            'likes' => $vlogVideo->likes,
            'category' => $vlogVideo->category,
            'tags' => $vlogVideo->tags,
            'featured' => $vlogVideo->featured,
            'status' => $vlogVideo->status,
            'published_at' => $vlogVideo->published_at?->format('Y-m-d'),
            'sort_order' => $vlogVideo->sort_order,
        ];

        return Inertia::render('admin/frontend/vlog/edit', [
            'vlogVideo' => $video,
            'categories' => $categories,
            'platforms' => $platforms,
        ]);
    }

    public function update(Request $request, VlogVideo $vlogVideo)
    {
        $categories = VlogVideo::getCategories();
        $platforms = VlogVideo::getPlatforms();

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'platform' => 'required|string|in:' . implode(',', array_keys(VlogVideo::getPlatforms())),
            'video_url' => 'required|url',
            'thumbnail_url' => 'nullable|url',
            'duration' => 'nullable|integer|min:1',
            'views' => 'nullable|integer|min:0',
            'likes' => 'nullable|integer|min:0',
            'category' => 'required|string|in:' . implode(',', array_keys(VlogVideo::getCategories())),
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50',
            'featured' => 'boolean',
            'status' => 'required|string|in:draft,published,archived',
            'published_at' => 'nullable|date',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        $vlogVideo->update($validated);
        return redirect()->route('admin.frontend.vlog.index')
            ->with('success', 'Vlog video updated successfully.');
    }

    public function destroy($id)
    {
        $vlogVideo = VlogVideo::withTrashed()->findOrFail($id);

        if ($vlogVideo->trashed()) {
            $vlogVideo->forceDelete();
        } else {
            $vlogVideo->delete();
        }
        return redirect()->route('admin.frontend.vlog.index')
            ->with('success', 'Vlog video deleted successfully.');
    }

    public function restore($id)
    {
        $vlogVideo = VlogVideo::withTrashed()->findOrFail($id);
        $vlogVideo->restore();

        return redirect()->route('admin.frontend.vlog.index')
            ->with('success', 'Vlog video restored successfully.');
    }

    public function forceDelete($id)
    {
        $vlogVideo = VlogVideo::withTrashed()->findOrFail($id);
        $vlogVideo->forceDelete();

        return redirect()->route('admin.frontend.vlog.index')
            ->with('success', 'Vlog video permanently deleted.');
    }
}
