<?php

namespace App\Http\Controllers\Admin\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Frontend\TournamentHighlight;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Validator;

class TournamentHighlightAdminController extends Controller
{
    /**
     * Display a listing of tournament highlights
     */
    public function index()
    {
        $highlights = TournamentHighlight::orderBy('sort_order')->get();
        
        return Inertia::render('admin/frontend/tournament-highlights/index', [
            'highlights' => $highlights
        ]);
    }

    /**
     * Show the form for creating a new tournament highlight
     */
    public function create()
    {
        return Inertia::render('admin/frontend/tournament-highlights/create');
    }

    /**
     * Store a newly created tournament highlight
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'icon' => 'required|string|max:50',
            'title' => 'required|string|max:255',
            'description' => 'required|string|max:500',
            'value' => 'required|string|max:100',
            'color' => 'required|string|max:100',
            'sort_order' => 'nullable|integer|min:0',
            'status' => 'nullable|string|max:50',
            'active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $highlight = TournamentHighlight::create($request->all());

        return redirect()->route('admin.frontend.tournament-highlights.index')
            ->with('success', 'Tournament highlight created successfully.');
    }

    /**
     * Show the form for editing the specified tournament highlight
     */
    public function edit(TournamentHighlight $tournamentHighlight)
    {
        return Inertia::render('admin/frontend/tournament-highlights/edit', [
            'highlight' => $tournamentHighlight
        ]);
    }

    /**
     * Update the specified tournament highlight
     */
    public function update(Request $request, TournamentHighlight $tournamentHighlight)
    {
        $validator = Validator::make($request->all(), [
            'icon' => 'required|string|max:50',
            'title' => 'required|string|max:255',
            'description' => 'required|string|max:500',
            'value' => 'required|string|max:100',
            'color' => 'required|string|max:100',
            'sort_order' => 'nullable|integer|min:0',
            'status' => 'nullable|string|max:50',
            'active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $tournamentHighlight->update($request->all());

        return redirect()->route('admin.frontend.tournament-highlights.index')
            ->with('success', 'Tournament highlight updated successfully.');
    }

    /**
     * Remove the specified tournament highlight
     */
    public function destroy(TournamentHighlight $tournamentHighlight)
    {
        $tournamentHighlight->delete();

        return redirect()->route('admin.frontend.tournament-highlights.index')
            ->with('success', 'Tournament highlight deleted successfully.');
    }

    /**
     * Toggle the active status of a tournament highlight
     */
    public function toggleStatus(TournamentHighlight $tournamentHighlight)
    {
        $tournamentHighlight->update([
            'active' => !$tournamentHighlight->active
        ]);

        return back()->with('success', 'Tournament highlight status updated successfully.');
    }

    /**
     * Update the sort order of tournament highlights
     */
    public function updateSortOrder(Request $request)
    {
        $request->validate([
            'highlights' => 'required|array',
            'highlights.*.id' => 'required|exists:frontend_tournament_highlights,id',
            'highlights.*.sort_order' => 'required|integer|min:0'
        ]);

        foreach ($request->highlights as $highlight) {
            TournamentHighlight::where('id', $highlight['id'])
                ->update(['sort_order' => $highlight['sort_order']]);
        }

        return response()->json(['message' => 'Sort order updated successfully']);
    }

    /**
     * Bulk update tournament highlights
     */
    public function bulkUpdate(Request $request)
    {
        $request->validate([
            'action' => 'required|in:activate,deactivate,delete',
            'ids' => 'required|array',
            'ids.*' => 'exists:frontend_tournament_highlights,id'
        ]);

        $highlights = TournamentHighlight::whereIn('id', $request->ids);

        switch ($request->action) {
            case 'activate':
                $highlights->update(['active' => true]);
                $message = 'Selected highlights activated successfully.';
                break;
            case 'deactivate':
                $highlights->update(['active' => false]);
                $message = 'Selected highlights deactivated successfully.';
                break;
            case 'delete':
                $highlights->delete();
                $message = 'Selected highlights deleted successfully.';
                break;
        }

        return back()->with('success', $message);
    }
} 