<?php

namespace App\Http\Controllers\Admin\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Frontend\Testimonial;
use Illuminate\Http\Request;
use Inertia\Inertia;

class TestimonialAdminController extends Controller
{
    public function index(Request $request)
    {
        $query = Testimonial::query();
        
        if ($request->filled('search')) {
            $query->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('role', 'like', '%' . $request->search . '%')
                  ->orWhere('team', 'like', '%' . $request->search . '%');
        }
        
        if ($request->filled('rating') && $request->rating !== 'all') {
            $query->where('rating', $request->rating);
        }
        
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }
        
        if ($request->filled('featured') && $request->featured !== 'all') {
            $query->where('featured', $request->featured === 'true');
        }
        
        $testimonials = $query->ordered()->paginate(10);
        
        return Inertia::render('admin/frontend/testimonials/index', [
            'testimonials' => $testimonials,
            'filters' => $request->only(['search', 'rating', 'status', 'featured'])
        ]);
    }

    public function create()
    {
        return Inertia::render('admin/frontend/testimonials/create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'role' => 'required|string|max:255',
            'team' => 'nullable|string|max:255',
            'image' => 'nullable|string',
            'rating' => 'required|integer|min:1|max:5',
            'quote' => 'required|string',
            'achievement' => 'nullable|string|max:500',
            'year' => 'nullable|integer|min:2020|max:' . (date('Y') + 1),
            'sort_order' => 'nullable|integer|min:0',
            'status' => 'required|in:draft,published,archived',
            'active' => 'boolean',
            'featured' => 'boolean'
        ]);

        // Set default values
        $validated['sort_order'] = $validated['sort_order'] ?? 0;
        $validated['year'] = $validated['year'] ?? date('Y');

        Testimonial::create($validated);

        return redirect()->route('admin.frontend.testimonials.index')
            ->with('success', 'Testimonial created successfully.');
    }

    public function show(Testimonial $testimonial)
    {
        return Inertia::render('admin/frontend/testimonials/show', [
            'testimonial' => $testimonial
        ]);
    }

    public function edit(Testimonial $testimonial)
    {
        return Inertia::render('admin/frontend/testimonials/edit', [
            'testimonial' => $testimonial
        ]);
    }

    public function update(Request $request, Testimonial $testimonial)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'role' => 'required|string|max:255',
            'team' => 'nullable|string|max:255',
            'image' => 'nullable|string',
            'rating' => 'required|integer|min:1|max:5',
            'quote' => 'required|string',
            'achievement' => 'nullable|string|max:500',
            'year' => 'nullable|integer|min:2020|max:' . (date('Y') + 1),
            'sort_order' => 'nullable|integer|min:0',
            'status' => 'required|in:draft,published,archived',
            'active' => 'boolean',
            'featured' => 'boolean'
        ]);

        $testimonial->update($validated);

        return redirect()->route('admin.frontend.testimonials.index')
            ->with('success', 'Testimonial updated successfully.');
    }

    public function destroy(Testimonial $testimonial)
    {
        // Delete the image file if it exists
        if ($testimonial->image && file_exists(public_path($testimonial->image))) {
            unlink(public_path($testimonial->image));
        }
        
        $testimonial->delete();

        return redirect()->route('admin.frontend.testimonials.index')
            ->with('success', 'Testimonial deleted successfully.');
    }
}
