<?php

namespace App\Http\Controllers\Admin\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Frontend\Sponsor;
use Illuminate\Http\Request;
use Inertia\Inertia;

class SponsorAdminController extends Controller
{
    public function index(Request $request)
    {
        $query = Sponsor::query();
        
        if ($request->filled('search')) {
            $query->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
        }
        
        if ($request->filled('category') && $request->category !== 'all') {
            $query->where('category', $request->category);
        }
        
        if ($request->filled('tier') && $request->tier !== 'all') {
            $query->where('tier', $request->tier);
        }
        
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }
        
        $sponsors = $query->ordered()->paginate(10);
        
        return Inertia::render('admin/frontend/sponsors/index', [
            'sponsors' => $sponsors,
            'filters' => $request->only(['search', 'category', 'tier', 'status']),
            'categories' => Sponsor::getCategories(),
            'tiers' => Sponsor::getTiers()
        ]);
    }

    public function create()
    {
        return Inertia::render('admin/frontend/sponsors/create', [
            'categories' => Sponsor::getCategories(),
            'tiers' => Sponsor::getTiers()
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'category' => 'required|string',
            'logo' => 'nullable|string',
            'description' => 'nullable|string',
            'tier' => 'required|in:platinum,gold,silver,bronze',
            'website_url' => 'nullable|url|max:255',
            'contact_email' => 'nullable|email|max:255',
            'contact_phone' => 'nullable|string|max:20',
            'sort_order' => 'nullable|integer|min:0',
            'status' => 'required|in:draft,published,archived',
            'active' => 'boolean'
        ]);

        // Set default values
        $validated['sort_order'] = $validated['sort_order'] ?? 0;

        Sponsor::create($validated);

        return redirect()->route('admin.frontend.sponsors.index')
            ->with('success', 'Sponsor created successfully.');
    }

    public function show(Sponsor $sponsor)
    {
        return Inertia::render('admin/frontend/sponsors/show', [
            'sponsor' => $sponsor
        ]);
    }

    public function edit(Sponsor $sponsor)
    {
        return Inertia::render('admin/frontend/sponsors/edit', [
            'sponsor' => $sponsor,
            'categories' => Sponsor::getCategories(),
            'tiers' => Sponsor::getTiers()
        ]);
    }

    public function update(Request $request, Sponsor $sponsor)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'category' => 'required|string',
            'logo' => 'nullable|string',
            'description' => 'nullable|string',
            'tier' => 'required|in:platinum,gold,silver,bronze',
            'website_url' => 'nullable|url|max:255',
            'contact_email' => 'nullable|email|max:255',
            'contact_phone' => 'nullable|string|max:20',
            'sort_order' => 'nullable|integer|min:0',
            'status' => 'required|in:draft,published,archived',
            'active' => 'boolean'
        ]);

        $sponsor->update($validated);

        return redirect()->route('admin.frontend.sponsors.index')
            ->with('success', 'Sponsor updated successfully.');
    }

    public function destroy(Sponsor $sponsor)
    {
        // Delete the logo file if it exists
        if ($sponsor->logo && file_exists(public_path($sponsor->logo))) {
            unlink(public_path($sponsor->logo));
        }
        
        $sponsor->delete();

        return redirect()->route('admin.frontend.sponsors.index')
            ->with('success', 'Sponsor deleted successfully.');
    }
} 