<?php

namespace App\Http\Controllers\Admin\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Frontend\NewsArticle;
use Illuminate\Http\Request;
use Inertia\Inertia;

class NewsArticleAdminController extends Controller
{
    public function index(Request $request)
    {
        $query = NewsArticle::query();
        
        if ($request->filled('search')) {
            $query->where('title', 'like', '%' . $request->search . '%');
        }
        
        if ($request->filled('category') && $request->category !== 'all') {
            $query->where('category', $request->category);
        }
        
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }
        
        $articles = $query->orderBy('created_at', 'desc')->paginate(10);
        
        return Inertia::render('admin/frontend/news-articles/index', [
            'articles' => $articles,
            'filters' => $request->only(['search', 'category', 'status']),
            'categories' => NewsArticle::getCategories()
        ]);
    }

    public function create()
    {
        return Inertia::render('admin/frontend/news-articles/create', [
            'categories' => NewsArticle::getCategories()
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'excerpt' => 'nullable|string|max:500',
            'content' => 'required|string',
            'category' => 'required|string',
            'author' => 'required|string|max:255',
            'author_image' => 'nullable|string',
            'date' => 'nullable|date',
            'read_time' => 'nullable|integer|min:1',
            'image' => 'nullable|string',
            'tags' => 'nullable|array',
            'featured' => 'boolean',
            'status' => 'required|in:draft,published,archived',
            'published_at' => 'nullable|date',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string|max:255'
        ]);

        // Set default values
        $validated['date'] = $validated['date'] ?? now();
        $validated['views'] = 0;
        $validated['likes'] = 0;

        NewsArticle::create($validated);

        return redirect()->route('admin.frontend.news-articles.index')
            ->with('success', 'News article created successfully.');
    }

    public function show(NewsArticle $article)
    {
        return Inertia::render('admin/frontend/news-articles/show', [
            'article' => $article
        ]);
    }

    public function edit(NewsArticle $article)
    {
        return Inertia::render('admin/frontend/news-articles/edit', [
            'article' => $article,
            'categories' => NewsArticle::getCategories()
        ]);       
    }

    public function update(Request $request, NewsArticle $article)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'excerpt' => 'nullable|string|max:500',
            'content' => 'required|string',
            'category' => 'required|string',
            'author' => 'required|string|max:255',
            'author_image' => 'nullable|string',
            'date' => 'nullable|date',
            'read_time' => 'nullable|integer|min:1',
            'image' => 'nullable|string',
            'tags' => 'nullable|array',
            'featured' => 'boolean',
            'status' => 'required|in:draft,published,archived',
            'published_at' => 'nullable|date',
            'meta_description' => 'nullable|string|max:160',
            'meta_keywords' => 'nullable|string|max:255'
        ]);

        $article->update($validated);

        return redirect()->route('admin.frontend.news-articles.index')
            ->with('success', 'News article updated successfully.');
    }

    public function destroy(NewsArticle $article)
    {
        $article->delete();

        return redirect()->route('admin.frontend.news-articles.index')
            ->with('success', 'News article deleted successfully.');
    }

    public function toggleFeatured(NewsArticle $article)
    {
        $article->update(['featured' => !$article->featured]);

        return redirect()->route('admin.frontend.news-articles.index')
            ->with('success', 'Article featured status updated successfully.');
    }
} 
