<?php

namespace App\Http\Controllers\Admin\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Frontend\HeroSlide;
use Illuminate\Http\Request;
use Inertia\Inertia;

class HeroSlideAdminController extends Controller
{
    public function index(Request $request)
    {
        $query = HeroSlide::query();
        
        if ($request->filled('search')) {
            $query->where('title', 'like', '%' . $request->search . '%');
        }
        
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }
        
        $slides = $query->orderBy('sort_order')->paginate(10);
        
        return Inertia::render('admin/frontend/hero-slides/index', [
            'slides' => $slides,
            'filters' => $request->only(['search', 'status'])
        ]);
    }

    public function create()
    {
        return Inertia::render('admin/frontend/hero-slides/create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:500',
            'bg_image' => 'nullable|string',
            'cta_primary' => 'nullable|string|max:100',
            'cta_secondary' => 'nullable|string|max:100',
            'cta_primary_url' => 'nullable|url|max:255',
            'cta_secondary_url' => 'nullable|url|max:255',
            'sort_order' => 'nullable|integer|min:0',
            'status' => 'required|in:draft,published,archived',
            'active' => 'boolean'
        ]);

        HeroSlide::create($validated);

        return redirect()->route('admin.frontend.hero-slides.index')
            ->with('success', 'Hero slide created successfully.');
    }

    public function edit(HeroSlide $slide)
    {
        return Inertia::render('admin/frontend/hero-slides/edit', [
            'slide' => $slide
        ]);
    }

    public function update(Request $request, HeroSlide $slide)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:500',
            'bg_image' => 'nullable|string',
            'cta_primary' => 'nullable|string|max:100',
            'cta_secondary' => 'nullable|string|max:100',
            'cta_primary_url' => 'nullable|url|max:255',
            'cta_secondary_url' => 'nullable|url|max:255',
            'sort_order' => 'nullable|integer|min:0',
            'status' => 'required|in:draft,published,archived',
            'active' => 'boolean'
        ]);

        $slide->update($validated);

        return redirect()->route('admin.frontend.hero-slides.index')
            ->with('success', 'Hero slide updated successfully.');
    }

    public function destroy(HeroSlide $slide)
    {
        $slide->delete();

        return redirect()->route('admin.frontend.hero-slides.index')
            ->with('success', 'Hero slide deleted successfully.');
    }
} 