<?php

namespace App\Http\Controllers\Admin\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Frontend\GalleryImage;
use Illuminate\Http\Request;
use Inertia\Inertia;

class GalleryImageAdminController extends Controller
{
    public function index(Request $request)
    {
        $query = GalleryImage::query();
        
        if ($request->filled('search')) {
            $query->where('title', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
        }
        
        if ($request->filled('category') && $request->category !== 'all') {
            $query->where('category', $request->category);
        }
        
        if ($request->filled('edition') && $request->edition !== 'all') {
            $query->where('edition', $request->edition);
        }
        
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }
        
        $images = $query->ordered()->paginate(12);
        
        return Inertia::render('admin/frontend/gallery/index', [
            'images' => $images,
            'filters' => $request->only(['search', 'category', 'edition', 'status']),
            'categories' => GalleryImage::getCategories()
        ]);
    }

    public function create()
    {
        return Inertia::render('admin/frontend/gallery/create', [
            'categories' => GalleryImage::getCategories(),
            'editions' => GalleryImage::getEditions()
        ]);
    }

    public function bulkCreate()
    {
        return Inertia::render('admin/frontend/gallery/bulk-create', [
            'categories' => GalleryImage::getCategories(),
            'editions' => GalleryImage::getEditions()
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'src' => 'required|string',
            'alt' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'required|string',
            'edition' => 'nullable|string|max:50',
            'height' => 'nullable|integer|min:100',
            'photographer' => 'nullable|string|max:255',
            'date_taken' => 'nullable|date',
            'status' => 'required|in:draft,published,archived',
            'sort_order' => 'nullable|integer|min:0',
            'featured' => 'boolean'
        ]);

        // Set default values
        $validated['sort_order'] = $validated['sort_order'] ?? 0;

        GalleryImage::create($validated);

        return redirect()->route('admin.frontend.gallery.index')
            ->with('success', 'Gallery image created successfully.');
    }

    public function bulkStore(Request $request)
    {
        $validated = $request->validate([
            'images' => 'required|array|min:1',
            'images.*.src' => 'required|string',
            'images.*.alt' => 'required|string|max:255',
            'images.*.title' => 'required|string|max:255',
            'images.*.description' => 'nullable|string',
            'images.*.category' => 'required|string',
            'images.*.edition' => 'nullable|string|max:50',
            'images.*.height' => 'nullable|integer|min:100',
            'images.*.photographer' => 'nullable|string|max:255',
            'images.*.date_taken' => 'nullable|date',
            'images.*.status' => 'required|in:draft,published,archived',
            'images.*.sort_order' => 'nullable|integer|min:0',
            'images.*.featured' => 'boolean'
        ]);

        $createdCount = 0;
        foreach ($validated['images'] as $imageData) {
            // Set default values
            $imageData['sort_order'] = $imageData['sort_order'] ?? 0;
            
            GalleryImage::create($imageData);
            $createdCount++;
        }

        return redirect()->route('admin.frontend.gallery.index')
            ->with('success', "Successfully created {$createdCount} gallery images.");
    }

    public function show(GalleryImage $image)
    {
        return Inertia::render('admin/frontend/gallery/show', [
            'image' => $image
        ]);
    }

    public function edit(GalleryImage $image)
    {
        return Inertia::render('admin/frontend/gallery/edit', [
            'image' => $image,
            'categories' => GalleryImage::getCategories()
        ]);
    }

    public function update(Request $request, GalleryImage $image)
    {
        $validated = $request->validate([
            'src' => 'required|string',
            'alt' => 'required|string|max:255',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'required|string',
            'edition' => 'nullable|string|max:50',
            'height' => 'nullable|integer|min:100',
            'photographer' => 'nullable|string|max:255',
            'date_taken' => 'nullable|date',
            'status' => 'required|in:draft,published,archived',
            'sort_order' => 'nullable|integer|min:0',
            'featured' => 'boolean'
        ]);

        $image->update($validated);

        return redirect()->route('admin.frontend.gallery.index')
            ->with('success', 'Gallery image updated successfully.');
    }

    public function destroy(GalleryImage $image)
    {
        // Delete the image file if it exists
        if ($image->src && file_exists(public_path($image->src))) {
            unlink(public_path($image->src));
        }
        
        $image->delete();

        return redirect()->route('admin.frontend.gallery.index')
            ->with('success', 'Gallery image deleted successfully.');
    }
}
