<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Faq;
use Illuminate\Http\Request;
use Inertia\Inertia;

class FaqController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Faq::query();

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('question', 'like', "%{$search}%")
                  ->orWhere('answer', 'like', "%{$search}%");
            });
        }

        // Category filter
        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        // Status filter
        if ($request->filled('status')) {
            $query->where('is_active', $request->status === 'active');
        }

        $faqs = $query->ordered()->paginate(15);

        return Inertia::render('admin/faqs/Index', [
            'faqs' => $faqs,
            'filters' => $request->only(['search', 'category', 'status']),
            'categories' => Faq::getCategories(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('admin/faqs/Create', [
            'categories' => Faq::getCategories(),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'category' => 'required|string|in:' . implode(',', array_keys(Faq::getCategories())),
            'question' => 'required|string|max:255',
            'answer' => 'required|string',
            'sort_order' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        Faq::create($validated);

        return redirect()->route('admin.faqs.index')
            ->with('success', 'FAQ created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Faq $faq)
    {
        return Inertia::render('admin/faqs/Show', [
            'faq' => $faq,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Faq $faq)
    {
        return Inertia::render('admin/faqs/Edit', [
            'faq' => $faq,
            'categories' => Faq::getCategories(),
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Faq $faq)
    {
        $validated = $request->validate([
            'category' => 'required|string|in:' . implode(',', array_keys(Faq::getCategories())),
            'question' => 'required|string|max:255',
            'answer' => 'required|string',
            'sort_order' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        $faq->update($validated);

        return redirect()->route('admin.faqs.index')
            ->with('success', 'FAQ updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Faq $faq)
    {
        $faq->delete();

        return redirect()->route('admin.faqs.index')
            ->with('success', 'FAQ deleted successfully.');
    }

    /**
     * Toggle FAQ active status
     */
    public function toggleStatus(Faq $faq)
    {
        $faq->update(['is_active' => !$faq->is_active]);

        return redirect()->route('admin.faqs.index')
            ->with('success', 'FAQ status updated successfully.');
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $validated = $request->validate([
            'action' => 'required|in:activate,deactivate,delete',
            'faq_ids' => 'required|array',
            'faq_ids.*' => 'exists:faqs,id',
        ]);

        $faqs = Faq::whereIn('id', $validated['faq_ids']);

        switch ($validated['action']) {
            case 'activate':
                $faqs->update(['is_active' => true]);
                $message = 'Selected FAQs activated successfully.';
                break;
            case 'deactivate':
                $faqs->update(['is_active' => false]);
                $message = 'Selected FAQs deactivated successfully.';
                break;
            case 'delete':
                $faqs->delete();
                $message = 'Selected FAQs deleted successfully.';
                break;
        }

        return redirect()->route('admin.faqs.index')
            ->with('success', $message);
    }
}
