<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\EmailCredential;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Inertia\Inertia;

class EmailSettingsController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:emails.view')->only(['index', 'show']);
        $this->middleware('permission:emails.create')->only(['create', 'store']);
        $this->middleware('permission:emails.edit')->only(['edit', 'update']);
        $this->middleware('permission:emails.delete')->only(['destroy']);
        $this->middleware('permission:emails.assign')->only(['assignUsers', 'removeUsers', 'updateUserPermissions']);
    }

    public function index(Request $request)
    {
        $query = EmailCredential::with('users');

        // Filtering
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('provider', 'like', "%{$search}%");
            });
        }

        if ($request->filled('provider') && $request->provider !== 'all') {
            $query->where('provider', $request->provider);
        }

        if ($request->filled('is_active') && $request->is_active !== 'all') {
            $query->where('is_active', $request->is_active === 'true');
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $emailCredentials = $query->paginate($request->get('per_page', 15))->withQueryString();

        // Get available providers for filter
        $providers = EmailCredential::distinct()->pluck('provider')->sort()->values();

        return Inertia::render('admin/settings/emails/index', [
            'emailCredentials' => $emailCredentials,
            'providers' => $providers,
            'filters' => $request->only(['search', 'provider', 'is_active', 'sort_by', 'sort_order']),
        ]);
    }

    public function create()
    {
        $providers = [
            'Gmail' => 'Gmail',
            'Outlook' => 'Outlook',
            'Yahoo' => 'Yahoo',
            'iCloud' => 'iCloud',
            'Exchange' => 'Exchange',
            'Other' => 'Other',
        ];

        return Inertia::render('admin/settings/emails/create', [
            'providers' => $providers,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:email_credentials,email',
            'provider' => 'required|string|max:255',
            'smtp_host' => 'required|string|max:255',
            'smtp_port' => 'required|integer|between:1,65535',
            'smtp_username' => 'required|string|max:255',
            'smtp_password' => 'required|string',
            'smtp_encryption' => 'required|string|in:tls,ssl,none',
            'is_active' => 'boolean',
            'instructions' => 'nullable|string',
            'metadata' => 'nullable|array',
        ]);

        $emailCredential = EmailCredential::create($validated);

        return redirect()->route('admin.settings.emails.show', $emailCredential)
                        ->with('success', 'Email credentials created successfully!');
    }

    public function show(EmailCredential $emailCredential)
    {
        $emailCredential->load('users');
        
        return Inertia::render('admin/settings/emails/show', [
            'emailCredential' => $emailCredential,
            'setupInstructions' => $emailCredential->getSetupInstructions(),
        ]);
    }

    public function edit(EmailCredential $emailCredential)
    {
        $providers = [
            'Gmail' => 'Gmail',
            'Outlook' => 'Outlook',
            'Yahoo' => 'Yahoo',
            'iCloud' => 'iCloud',
            'Exchange' => 'Exchange',
            'Other' => 'Other',
        ];

        return Inertia::render('admin/settings/emails/edit', [
            'emailCredential' => $emailCredential,
            'providers' => $providers,
        ]);
    }

    public function update(Request $request, EmailCredential $emailCredential)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:email_credentials,email,' . $emailCredential->id,
            'provider' => 'required|string|max:255',
            'smtp_host' => 'required|string|max:255',
            'smtp_port' => 'required|integer|between:1,65535',
            'smtp_username' => 'required|string|max:255',
            'smtp_password' => 'nullable|string',
            'smtp_encryption' => 'required|string|in:tls,ssl,none',
            'is_active' => 'boolean',
            'instructions' => 'nullable|string',
            'metadata' => 'nullable|array',
        ]);

        // Only update password if provided
        if (empty($validated['smtp_password'])) {
            unset($validated['smtp_password']);
        }

        $emailCredential->update($validated);

        return redirect()->route('admin.settings.emails.show', $emailCredential)
                        ->with('success', 'Email credentials updated successfully!');
    }

    public function destroy(EmailCredential $emailCredential)
    {
        // Check if email has users assigned
        if ($emailCredential->users()->count() > 0) {
            return redirect()->back()->with('error', 'Cannot delete email credentials with assigned users!');
        }

        $emailCredential->delete();

        return redirect()->route('admin.settings.emails.index')
                        ->with('success', 'Email credentials deleted successfully!');
    }

    public function toggleStatus(EmailCredential $emailCredential)
    {
        $emailCredential->update(['is_active' => !$emailCredential->is_active]);

        $status = $emailCredential->is_active ? 'activated' : 'deactivated';
        return redirect()->back()->with('success', "Email credentials {$status} successfully!");
    }

    public function assignUsers(Request $request, EmailCredential $emailCredential)
    {
        $validated = $request->validate([
            'user_ids' => 'required|array',
            'user_ids.*' => 'exists:users,id',
            'permissions' => 'nullable|array',
        ]);

        foreach ($validated['user_ids'] as $userId) {
            $permissions = $validated['permissions'][$userId] ?? [
                'can_send' => true,
                'can_receive' => true,
                'permissions' => null,
            ];

            $emailCredential->grantUserAccess(User::find($userId), $permissions);
        }

        return redirect()->back()->with('success', 'Users assigned to email credentials successfully!');
    }

    public function removeUsers(Request $request, EmailCredential $emailCredential)
    {
        $validated = $request->validate([
            'user_ids' => 'required|array',
            'user_ids.*' => 'exists:users,id',
        ]);

        foreach ($validated['user_ids'] as $userId) {
            $emailCredential->revokeUserAccess(User::find($userId));
        }

        return redirect()->back()->with('success', 'Users removed from email credentials successfully!');
    }

    public function updateUserPermissions(Request $request, EmailCredential $emailCredential)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'permissions' => 'required|array',
        ]);

        $emailCredential->updateUserPermissions(
            User::find($validated['user_id']),
            $validated['permissions']
        );

        return redirect()->back()->with('success', 'User permissions updated successfully!');
    }

    public function getAvailableUsers(Request $request, EmailCredential $emailCredential)
    {
        $search = $request->get('search', '');
        
        $users = User::whereNotIn('id', $emailCredential->users()->pluck('user_id'))
                    ->where(function($query) use ($search) {
                        $query->where('name', 'like', "%{$search}%")
                              ->orWhere('email', 'like', "%{$search}%");
                    })
                    ->select('id', 'name', 'email')
                    ->limit(20)
                    ->get();

        return response()->json($users);
    }

    public function getAssignedUsers(EmailCredential $emailCredential)
    {
        $users = $emailCredential->users()
                                ->select('users.id', 'users.name', 'users.email')
                                ->get();

        return response()->json($users);
    }

    public function testConnection(EmailCredential $emailCredential)
    {
        try {
            // Test SMTP connection
            $transport = new \Swift_SmtpTransport(
                $emailCredential->smtp_host,
                $emailCredential->smtp_port,
                $emailCredential->smtp_encryption
            );
            
            $transport->setUsername($emailCredential->smtp_username);
            $transport->setPassword($emailCredential->smtp_password);
            
            $mailer = new \Swift_Mailer($transport);
            $mailer->getTransport()->start();
            
            return response()->json([
                'success' => true,
                'message' => 'SMTP connection successful!',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'SMTP connection failed: ' . $e->getMessage(),
            ], 400);
        }
    }

    public function bulkUpdate(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:email_credentials,id',
            'action' => 'required|string|in:activate,deactivate,delete',
        ]);

        $emailCredentials = EmailCredential::whereIn('id', $validated['ids'])->get();
        $updated = 0;
        $errors = [];

        foreach ($emailCredentials as $emailCredential) {
            try {
                switch ($validated['action']) {
                    case 'activate':
                        $emailCredential->update(['is_active' => true]);
                        $updated++;
                        break;
                    case 'deactivate':
                        $emailCredential->update(['is_active' => false]);
                        $updated++;
                        break;
                    case 'delete':
                        // Check if email has users assigned
                        if ($emailCredential->users()->count() > 0) {
                            $errors[] = "Cannot delete {$emailCredential->name} - has assigned users";
                            break;
                        }
                        $emailCredential->delete();
                        $updated++;
                        break;
                }
            } catch (\Exception $e) {
                $errors[] = "Error processing {$emailCredential->name}: " . $e->getMessage();
            }
        }

        $message = "Successfully {$validated['action']}d {$updated} email account(s)";
        if (!empty($errors)) {
            $message .= ". Errors: " . implode(', ', $errors);
        }

        return redirect()->back()->with('success', $message);
    }

    public function bulkDelete(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:email_credentials,id',
        ]);

        $emailCredentials = EmailCredential::whereIn('id', $validated['ids'])->get();
        $deleted = 0;
        $errors = [];

        foreach ($emailCredentials as $emailCredential) {
            try {
                // Check if email has users assigned
                if ($emailCredential->users()->count() > 0) {
                    $errors[] = "Cannot delete {$emailCredential->name} - has assigned users";
                    continue;
                }
                $emailCredential->delete();
                $deleted++;
            } catch (\Exception $e) {
                $errors[] = "Error deleting {$emailCredential->name}: " . $e->getMessage();
            }
        }

        $message = "Successfully deleted {$deleted} email account(s)";
        if (!empty($errors)) {
            $message .= ". Errors: " . implode(', ', $errors);
        }

        return redirect()->back()->with('success', $message);
    }
}
