<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Tournament;
use App\Models\Player;
use App\Models\Team;
use App\Models\GameMatch;
use App\Models\User;
use Illuminate\Http\Request;
use Inertia\Inertia;

class DashboardController extends Controller
{
    public function __construct()
    {
        // Allow all authenticated users to access dashboard
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $user = $request->user();
        
        // Get user permissions
        $userPermissions = [];
        $userRoles = [];
        
        if ($user) {
            // Get all user roles
            $userRoles = $user->roles()->pluck('name')->toArray();
            
            // Get all user permissions
            $userPermissions = $user->getAllPermissions()->pluck('name')->toArray();
        }

        // Base statistics that all users can see
        $stats = [
            'total_tournaments' => Tournament::count(),
            'active_tournaments' => Tournament::where('status', 'active')->count(),
        ];

        // Add role-specific statistics based on data scoping
        if ($this->canViewPlayerData($userRoles, $userPermissions)) {
            $stats['total_players'] = Player::count();
        }

        if ($this->canViewTeamData($userRoles, $userPermissions)) {
            $stats['total_teams'] = Team::count();
        }

        if ($this->canViewMatchData($userRoles, $userPermissions)) {
            $stats['total_matches'] = GameMatch::count();
            $stats['completed_matches'] = GameMatch::where('status', 'completed')->count();
        }

        // Only show user count to admins
        if (in_array('admin', $userRoles) || in_array('super_admin', $userRoles)) {
            $stats['total_users'] = User::count();
        }

        // Get recent activity based on permissions and data scoping
        $recentActivity = [];
        
        if ($this->canViewTournamentData($userRoles, $userPermissions)) {
            $recentActivity['recent_tournaments'] = Tournament::latest()->limit(3)->get();
        }

        if ($this->canViewPlayerData($userRoles, $userPermissions)) {
            $recentActivity['recent_players'] = Player::with('user')->latest()->limit(3)->get();
        }

        if ($this->canViewTeamData($userRoles, $userPermissions)) {
            $recentActivity['recent_teams'] = Team::latest()->limit(3)->get();
        }

        // Get recent matches with role-based scoping
        $recentMatches = $this->getScopedRecentMatches($user, $userRoles, $userPermissions);
        $stats['recent_matches'] = $recentMatches;

        // Get teams for team managers
        $teams = [];
        if (in_array('team_manager', $userRoles)) {
            $teams = $user->managedTeams()->select('id', 'name', 'description', 'logo', 'created_at')->get();
        }

        return Inertia::render('admin/Dashboard', [
            'stats' => $stats,
            'recentActivity' => $recentActivity,
            'teams' => $teams,
            'userPermissions' => $userPermissions,
            'userRoles' => $userRoles,
            'user' => $user ? [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'roles' => $userRoles,
                'permissions' => $userPermissions,
            ] : null,
        ]);
    }

    /**
     * Check if user can view tournament data
     */
    private function canViewTournamentData(array $userRoles, array $userPermissions): bool
    {
        return in_array('admin', $userRoles) || 
               in_array('super_admin', $userRoles) || 
               in_array('organizer', $userRoles) ||
               in_array('referee', $userRoles) ||
               in_array('team_manager', $userRoles) ||
               in_array('tournaments.view', $userPermissions) ||
               in_array('tournaments.manage', $userPermissions);
    }

    /**
     * Check if user can view player data
     */
    private function canViewPlayerData(array $userRoles, array $userPermissions): bool
    {
        return in_array('admin', $userRoles) || 
               in_array('super_admin', $userRoles) || 
               in_array('organizer', $userRoles) ||
               in_array('team_manager', $userRoles) ||
               in_array('players.view', $userPermissions) ||
               in_array('players.manage', $userPermissions);
    }

    /**
     * Check if user can view team data
     */
    private function canViewTeamData(array $userRoles, array $userPermissions): bool
    {
        return in_array('admin', $userRoles) || 
               in_array('super_admin', $userRoles) || 
               in_array('organizer', $userRoles) ||
               in_array('team_manager', $userRoles) ||
               in_array('teams.view', $userPermissions) ||
               in_array('teams.manage', $userPermissions);
    }

    /**
     * Check if user can view match data
     */
    private function canViewMatchData(array $userRoles, array $userPermissions): bool
    {
        return in_array('admin', $userRoles) || 
               in_array('super_admin', $userRoles) || 
               in_array('organizer', $userRoles) ||
               in_array('referee', $userRoles) ||
               in_array('team_manager', $userRoles) ||
               in_array('matches.view', $userPermissions) ||
               in_array('matches.manage', $userPermissions);
    }

    /**
     * Get scoped recent matches based on user role and permissions
     */
    private function getScopedRecentMatches($user, array $userRoles, array $userPermissions): array
    {
        $query = GameMatch::with(['tournament', 'team1', 'team2']);

        // Apply role-based data scoping
        if (in_array('player', $userRoles)) {
            // Players can only see matches they're involved in
            $player = Player::where('user_id', $user->id)->first();
            if ($player) {
                $query->where(function($q) use ($player) {
                    $q->whereHas('team1.players', function($subQ) use ($player) {
                        $subQ->where('user_id', $player->id);
                    })->orWhereHas('team2.players', function($subQ) use ($player) {
                        $subQ->where('user_id', $player->id);
                    });
                });
            }
        } elseif (in_array('team_manager', $userRoles)) {
            // Team managers can see matches for teams they manage
            $managedTeamIds = Team::where('captain_id', $user->id)->pluck('id');
            $query->where(function($q) use ($managedTeamIds) {
                $q->whereIn('team1_id', $managedTeamIds)
                  ->orWhereIn('team2_id', $managedTeamIds);
            });
        } elseif (in_array('referee', $userRoles)) {
            // Referees can see matches they're assigned to
            $query->where('referee_id', $user->id);
        } elseif (in_array('organizer', $userRoles)) {
            // Organizers can see matches in tournaments they manage
            $managedTournamentIds = Tournament::where('organizer', $user->id)->pluck('id');
            $query->whereIn('tournament_id', $managedTournamentIds);
        }
        // Admins and super_admins can see all matches (no additional filtering)

        return $query->latest()->limit(5)->get()->toArray();
    }
}