<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Bracket;
use App\Models\Tournament;
use App\Models\Team;
use App\Models\Player;
use App\Services\TournamentBracketService;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class BracketAdminController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:brackets.view')->only(['index', 'show']);
        $this->middleware('permission:brackets.create')->only(['store', 'generateBracket']);
        $this->middleware('permission:brackets.edit')->only(['edit', 'update', 'advanceRound']);
        $this->middleware('permission:brackets.delete')->only(['destroy', 'destroyMultiple']);
    }

    public function index(Request $request)
    {
        $query = Bracket::with(['tournament', 'gameMatches']);
        
        if ($request->filled('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }
        if ($request->filled('type') && $request->type !== 'all') {
            $query->where('type', $request->type);
        }
        if ($request->filled('tournament_id') && $request->tournament_id !== 'all') {
            $query->where('tournament_id', $request->tournament_id);
        }
        
        $brackets = $query->orderByDesc('id')->paginate(10)->withQueryString();
        $tournaments = Tournament::select('id', 'name', 'type', 'status')->get();
        
        return Inertia::render('admin/brackets/index', [
            'brackets' => $brackets,
            'filters' => $request->only(['search', 'type', 'tournament_id']),
            'tournaments' => $tournaments,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:winners,losers,final,quarter-finals,semi-finals,third-place',
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'rounds' => 'nullable|integer',
        ]);
        
        $bracket = Bracket::create($validated);
        
        return redirect()->route('admin.brackets.index')->with('success', 'Bracket created!');
    }

    public function show(Bracket $bracket)
    {
        $bracket->load(['tournament', 'gameMatches.player1', 'gameMatches.player2', 'gameMatches.team1', 'gameMatches.team2']);
        
        $tournament = $bracket->tournament;
        $matches = $bracket->gameMatches;
        
        // Get participants based on tournament format
        $participants = [];
        if ($tournament->format === 'team') {
            $teams = $tournament->teams()->wherePivot('status', 'active')->get();
            $participants = $teams->map(function($team) {
                return [
                    'id' => $team->id,
                    'name' => $team->name,
                    'type' => 'team',
                    'rating' => $team->stats['rating'] ?? 1000,
                    'ranking' => $team->stats['ranking'] ?? 999,
                ];
            })->toArray();
        } else {
            $players = Player::where('tournament_id', $tournament->id)->get();
            $participants = $players->map(function($player) {
                return [
                    'id' => $player->id,
                    'name' => $player->display_name,
                    'type' => 'player',
                    'rating' => $player->rating ?? 1000,
                    'ranking' => $player->ranking ?? 999,
                ];
            })->toArray();
        }
        
        return Inertia::render('admin/brackets/show', [
            'bracket' => $bracket,
            'tournament' => $tournament,
            'matches' => $matches,
            'participants' => $participants,
            'stats' => [
                'total_matches' => $matches->count(),
                'completed_matches' => $matches->where('status', 'completed')->count(),
                'scheduled_matches' => $matches->where('status', 'scheduled')->count(),
                'in_progress_matches' => $matches->where('status', 'in_progress')->count(),
            ],
        ]);
    }

    public function generateBracket(Request $request, Tournament $tournament)
    {
        $validated = $request->validate([
            'bracket_name' => 'required|string|max:255',
            'bracket_type' => 'required|in:winners,losers,final',
            'seeding_method' => 'required|in:random,ranking,rating,balanced',
            'participants' => 'required|array|min:2',
            'participants.*.id' => 'required|integer',
            'participants.*.type' => 'required|in:player,team',
        ]);

        try {
            DB::beginTransaction();

            // Create bracket
            $bracket = Bracket::create([
                'tournament_id' => $tournament->id,
                'name' => $validated['bracket_name'],
                'type' => $validated['bracket_type'],
                'rounds' => 0, // Will be set by the service
                'structure' => [],
                'seeding' => [],
                'is_active' => true,
            ]);

            // Generate bracket using enhanced service
            $bracketService = new TournamentBracketService($tournament);
            $generatedBracket = $bracketService->generateBracket($validated['participants'], $validated['bracket_type']);

            DB::commit();

            return redirect()->route('admin.brackets.show', $generatedBracket)
                ->with('success', 'Bracket generated successfully!');

        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Bracket generation failed: ' . $e->getMessage());
            
            return back()->withErrors(['generation' => 'Bracket generation failed: ' . $e->getMessage()]);
        }
    }

    public function advanceRound(Request $request, Bracket $bracket)
    {
        $validated = $request->validate([
            'current_round' => 'required|integer|min:1',
        ]);

        try {
            $bracketService = new TournamentBracketService($bracket->tournament);
            $result = $bracketService->advanceRound($bracket, $validated['current_round']);

            return back()->with('success', 
                "Round {$result['round']} created with {$result['matches_created']} matches!");

        } catch (\Exception $e) {
            Log::error('Round advancement failed: ' . $e->getMessage());
            return back()->withErrors(['advancement' => 'Round advancement failed: ' . $e->getMessage()]);
        }
    }

    public function edit(Bracket $bracket)
    {
        $tournaments = Tournament::select('id', 'name', 'type', 'status')->get();
        return Inertia::render('admin/brackets/edit', [
            'bracket' => $bracket,
            'tournaments' => $tournaments,
        ]);
    }

    public function update(Request $request, Bracket $bracket)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:winners,losers,final,quarter-finals,semi-finals,third-place',
            'tournament_id' => 'required|integer|exists:tournaments,id',
            'rounds' => 'nullable|integer',
            'is_active' => 'boolean',
        ]);
        
        $bracket->update($validated);
        return redirect()->route('admin.brackets.index')->with('success', 'Bracket updated!');
    }

    public function destroy(Bracket $bracket)
    {
        // Delete associated matches first
        $bracket->gameMatches()->delete();
        $bracket->delete();
        
        return redirect()->route('admin.brackets.index')->with('success', 'Bracket deleted!');
    }

    public function destroyMultiple(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:brackets,id'
        ]);
        
        // Delete associated matches first
        foreach ($validated['ids'] as $bracketId) {
            $bracket = Bracket::find($bracketId);
            if ($bracket) {
                $bracket->gameMatches()->delete();
                $bracket->delete();
            }
        }
        
        return redirect()->route('admin.brackets.index')->with('success', 'Brackets deleted!');
    }

    /**
     * Get tournament participants for bracket generation
     */
    public function getParticipants(Tournament $tournament)
    {
        $participants = [];
        
        if ($tournament->format === 'team') {
            $teams = $tournament->teams()->wherePivot('status', 'active')->get();
            $participants = $teams->map(function($team) {
                return [
                    'id' => $team->id,
                    'name' => $team->name,
                    'type' => 'team',
                    'rating' => $team->stats['rating'] ?? 1000,
                    'ranking' => $team->stats['ranking'] ?? 999,
                ];
            })->toArray();
        } else {
            $players = Player::where('tournament_id', $tournament->id)->get();
            $participants = $players->map(function($player) {
                return [
                    'id' => $player->id,
                    'name' => $player->display_name,
                    'type' => 'player',
                    'rating' => $player->rating ?? 1000,
                    'ranking' => $player->ranking ?? 999,
                ];
            })->toArray();
        }

        return response()->json([
            'participants' => $participants,
            'tournament' => [
                'id' => $tournament->id,
                'name' => $tournament->name,
                'format' => $tournament->format,
                'type' => $tournament->type,
            ],
        ]);
    }
} 