<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Tournament;
use App\Models\Player;
use App\Models\Team;
use App\Models\GameMatch;
use App\Models\User;
use Illuminate\Http\Request;
use Inertia\Inertia;

class AnalyticsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $user = $request->user();
        $userRoles = $user->roles()->pluck('name')->toArray();
        $userPermissions = $user->getAllPermissions()->pluck('name')->toArray();

        // Get role-specific analytics
        $analytics = $this->getRoleBasedAnalytics($user, $userRoles, $userPermissions);

        return Inertia::render('admin/Analytics', [
            'analytics' => $analytics,
            'userRoles' => $userRoles,
            'userPermissions' => $userPermissions,
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
            ],
        ]);
    }

    private function getRoleBasedAnalytics($user, array $userRoles, array $userPermissions)
    {
        $analytics = [];

        // Admin Analytics
        if (in_array('admin', $userRoles) || in_array('super_admin', $userRoles)) {
            $analytics['system'] = [
                'total_users' => User::count(),
                'total_tournaments' => Tournament::count(),
                'active_tournaments' => Tournament::where('status', 'active')->count(),
                'total_matches' => GameMatch::count(),
                'completed_matches' => GameMatch::where('status', 'completed')->count(),
            ];
        }

        // Organizer Analytics
        if (in_array('organizer', $userRoles)) {
            $analytics['tournaments'] = [
                'total_tournaments' => Tournament::where('organizer_id', $user->id)->count(),
                'active_tournaments' => Tournament::where('organizer_id', $user->id)->where('status', 'active')->count(),
                'total_players' => Player::whereHas('teams', function($q) use ($user) {
                    $q->whereHas('tournament', function($subQ) use ($user) {
                        $subQ->where('organizer_id', $user->id);
                    });
                })->count(),
            ];
        }

        // Referee Analytics
        if (in_array('referee', $userRoles)) {
            $analytics['matches'] = [
                'total_matches_officiated' => GameMatch::where('referee_id', $user->id)->count(),
                'completed_matches' => GameMatch::where('referee_id', $user->id)->where('status', 'completed')->count(),
                'pending_matches' => GameMatch::where('referee_id', $user->id)->where('status', 'pending')->count(),
            ];
        }

        // Player Analytics
        if (in_array('player', $userRoles)) {
            $player = Player::where('user_id', $user->id)->first();
            if ($player) {
                $matches = GameMatch::where(function($q) use ($player) {
                    $q->whereHas('team1.players', function($subQ) use ($player) {
                        $subQ->where('player_id', $player->id);
                    })->orWhereHas('team2.players', function($subQ) use ($player) {
                        $subQ->where('player_id', $player->id);
                    });
                })->where('status', 'completed');

                $totalMatches = $matches->count();
                $wins = $this->calculatePlayerWins($player, $matches);

                $analytics['personal'] = [
                    'total_matches' => $totalMatches,
                    'wins' => $wins,
                    'losses' => $totalMatches - $wins,
                    'win_rate' => $totalMatches > 0 ? round(($wins / $totalMatches) * 100, 2) : 0,
                    'tournaments_participated' => Tournament::whereHas('teams.players', function($q) use ($player) {
                        $q->where('player_id', $player->id);
                    })->count(),
                ];
            }
        }

        // General Analytics for all users
        $analytics['general'] = [
            'total_tournaments' => Tournament::count(),
            'active_tournaments' => Tournament::where('status', 'active')->count(),
            'total_matches' => GameMatch::count(),
        ];

        return $analytics;
    }

    private function calculatePlayerWins($player, $matches)
    {
        return $matches->get()->filter(function($match) use ($player) {
            $playerTeam = $match->team1->players->contains('player_id', $player->id) ? $match->team1 : $match->team2;
            
            if ($match->team1->id === $playerTeam->id) {
                return $match->score_team1 > $match->score_team2;
            } else {
                return $match->score_team2 > $match->score_team1;
            }
        })->count();
    }
}