<?php

namespace App\Helpers;

use Illuminate\Support\Facades\File;

class StorageHelper
{
    /**
     * Setup public storage directory
     */
    public static function setupPublicStorage()
    {
        $publicStoragePath = public_path('storage');
        $storageAppPublicPath = storage_path('app/public');

        // Create public storage directory if it doesn't exist
        if (!is_dir($publicStoragePath)) {
            File::makeDirectory($publicStoragePath, 0755, true);
        }

        // Create storage/app/public directory if it doesn't exist
        if (!is_dir($storageAppPublicPath)) {
            File::makeDirectory($storageAppPublicPath, 0755, true);
        }

        // Set proper permissions
        chmod($publicStoragePath, 0755);
        chmod($storageAppPublicPath, 0755);

        return [
            'public_storage_created' => is_dir($publicStoragePath),
            'storage_app_public_created' => is_dir($storageAppPublicPath),
            'public_storage_writable' => is_writable($publicStoragePath),
            'storage_app_public_writable' => is_writable($storageAppPublicPath),
        ];
    }

    /**
     * Get storage status
     */
    public static function getStorageStatus()
    {
        $publicStoragePath = public_path('storage');
        $storageAppPublicPath = storage_path('app/public');
        $storagePath = storage_path();

        return [
            'public_storage_exists' => is_dir($publicStoragePath),
            'public_storage_writable' => is_writable($publicStoragePath),
            'storage_app_public_exists' => is_dir($storageAppPublicPath),
            'storage_app_public_writable' => is_writable($storageAppPublicPath),
            'storage_writable' => is_writable($storagePath),
            'public_storage_path' => $publicStoragePath,
            'storage_app_public_path' => $storageAppPublicPath,
        ];
    }

    /**
     * Copy files from storage/app/public to public/storage
     */
    public static function syncPublicStorage()
    {
        $sourcePath = storage_path('app/public');
        $destinationPath = public_path('storage');

        if (!is_dir($sourcePath)) {
            return false;
        }

        if (!is_dir($destinationPath)) {
            File::makeDirectory($destinationPath, 0755, true);
        }

        // Copy all files from storage/app/public to public/storage
        $files = File::allFiles($sourcePath);
        
        foreach ($files as $file) {
            $relativePath = $file->getRelativePathname();
            $destinationFile = $destinationPath . '/' . $relativePath;
            $destinationDir = dirname($destinationFile);
            
            if (!is_dir($destinationDir)) {
                File::makeDirectory($destinationDir, 0755, true);
            }
            
            File::copy($file->getPathname(), $destinationFile);
        }

        return true;
    }

    /**
     * Clean up old files in public storage that don't exist in storage/app/public
     */
    public static function cleanupPublicStorage()
    {
        $sourcePath = storage_path('app/public');
        $destinationPath = public_path('storage');

        if (!is_dir($sourcePath) || !is_dir($destinationPath)) {
            return false;
        }

        $sourceFiles = collect(File::allFiles($sourcePath))->map(function ($file) use ($sourcePath) {
            return str_replace($sourcePath . '/', '', $file->getPathname());
        })->toArray();

        $destinationFiles = collect(File::allFiles($destinationPath))->map(function ($file) use ($destinationPath) {
            return str_replace($destinationPath . '/', '', $file->getPathname());
        })->toArray();

        // Remove files in destination that don't exist in source
        foreach ($destinationFiles as $file) {
            if (!in_array($file, $sourceFiles)) {
                File::delete($destinationPath . '/' . $file);
            }
        }

        return true;
    }
}