<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;

class SettingsValidator
{
    /**
     * Validate database settings and provide recommendations
     */
    public static function validateDatabaseSettings()
    {
        $issues = [];
        $recommendations = [];

        $connection = config('database.default');
        $config = config("database.connections.{$connection}");

        if (!$config) {
            $issues[] = "Database connection '{$connection}' is not configured.";
            $recommendations[] = "Check your database configuration in config/database.php";
        }

        if ($connection === 'mysql' || $connection === 'pgsql') {
            if (empty($config['host'])) {
                $issues[] = "Database host is not configured for {$connection}.";
            }
            if (empty($config['database'])) {
                $issues[] = "Database name is not configured for {$connection}.";
            }
        }

        if ($connection === 'sqlite') {
            $database = $config['database'] ?? database_path('database.sqlite');
            if (!File::exists($database)) {
                $issues[] = "SQLite database file does not exist: {$database}";
                $recommendations[] = "Run 'php artisan migrate:install' to create the database";
            }
        }

        return [
            'issues' => $issues,
            'recommendations' => $recommendations,
        ];
    }

    /**
     * Validate mail settings and provide recommendations
     */
    public static function validateMailSettings()
    {
        $issues = [];
        $recommendations = [];

        $driver = config('mail.default');
        $config = config("mail.mailers.{$driver}");

        if (!$config) {
            $issues[] = "Mail driver '{$driver}' is not configured.";
            $recommendations[] = "Check your mail configuration in config/mail.php";
        }

        if ($driver === 'smtp') {
            if (empty($config['host'])) {
                $issues[] = "SMTP host is not configured.";
            }
            if (empty($config['port'])) {
                $issues[] = "SMTP port is not configured.";
            }
            if (empty($config['username'])) {
                $issues[] = "SMTP username is not configured.";
            }
            if (empty($config['password'])) {
                $issues[] = "SMTP password is not configured.";
            }
        }

        if (empty(config('mail.from.address'))) {
            $issues[] = "Mail from address is not configured.";
            $recommendations[] = "Set MAIL_FROM_ADDRESS in your .env file";
        }

        if (empty(config('mail.from.name'))) {
            $issues[] = "Mail from name is not configured.";
            $recommendations[] = "Set MAIL_FROM_NAME in your .env file";
        }

        return [
            'issues' => $issues,
            'recommendations' => $recommendations,
        ];
    }

    /**
     * Validate cache settings and provide recommendations
     */
    public static function validateCacheSettings()
    {
        $issues = [];
        $recommendations = [];

        $driver = config('cache.default');
        $config = config("cache.stores.{$driver}");

        if (!$config) {
            $issues[] = "Cache driver '{$driver}' is not configured.";
            $recommendations[] = "Check your cache configuration in config/cache.php";
        }

        if ($driver === 'redis') {
            if (!extension_loaded('redis')) {
                $issues[] = "Redis PHP extension is not installed.";
                $recommendations[] = "Install Redis PHP extension: sudo apt-get install php-redis";
            }
        }

        if ($driver === 'memcached') {
            if (!extension_loaded('memcached')) {
                $issues[] = "Memcached PHP extension is not installed.";
                $recommendations[] = "Install Memcached PHP extension: sudo apt-get install php-memcached";
            }
        }

        return [
            'issues' => $issues,
            'recommendations' => $recommendations,
        ];
    }

    /**
     * Validate queue settings and provide recommendations
     */
    public static function validateQueueSettings()
    {
        $issues = [];
        $recommendations = [];

        $connection = config('queue.default');
        $config = config("queue.connections.{$connection}");

        if (!$config) {
            $issues[] = "Queue connection '{$connection}' is not configured.";
            $recommendations[] = "Check your queue configuration in config/queue.php";
        }

        if ($connection === 'database') {
            // Check if jobs table exists
            try {
                \DB::table('jobs')->count();
            } catch (\Exception $e) {
                $issues[] = "Queue jobs table does not exist.";
                $recommendations[] = "Run 'php artisan queue:table' and 'php artisan migrate'";
            }
        }

        if ($connection === 'redis') {
            if (!extension_loaded('redis')) {
                $issues[] = "Redis PHP extension is not installed.";
                $recommendations[] = "Install Redis PHP extension: sudo apt-get install php-redis";
            }
        }

        return [
            'issues' => $issues,
            'recommendations' => $recommendations,
        ];
    }

    /**
     * Validate storage settings and provide recommendations
     */
    public static function validateStorageSettings()
    {
        $issues = [];
        $recommendations = [];

        $storageStatus = \App\Helpers\StorageHelper::getStorageStatus();

        // Check if public storage directory exists and is writable
        if (!$storageStatus['public_storage_exists']) {
            $issues[] = "Public storage directory does not exist.";
            $recommendations[] = "The system will automatically create the directory when needed.";
        } elseif (!$storageStatus['public_storage_writable']) {
            $issues[] = "Public storage directory is not writable.";
            $recommendations[] = "Set proper permissions: chmod -R 775 public/storage";
        }

        // Check storage directory permissions
        if (!$storageStatus['storage_writable']) {
            $issues[] = "Storage directory is not writable.";
            $recommendations[] = "Set proper permissions: chmod -R 775 storage/";
        }

        // Check if storage/app/public exists
        if (!$storageStatus['storage_app_public_exists']) {
            $issues[] = "Storage app/public directory does not exist.";
            $recommendations[] = "The system will automatically create the directory when needed.";
        } elseif (!$storageStatus['storage_app_public_writable']) {
            $issues[] = "Storage app/public directory is not writable.";
            $recommendations[] = "Set proper permissions: chmod -R 775 storage/app/public";
        }

        // Check filesystem configuration
        $filesystemDisk = config('filesystems.default');
        if ($filesystemDisk !== 'local' && $filesystemDisk !== 'public') {
            $recommendations[] = "Consider using 'local' or 'public' disk for better compatibility.";
        }

        return [
            'issues' => $issues,
            'recommendations' => $recommendations,
        ];
    }

    /**
     * Get all validation results
     */
    public static function getAllValidations()
    {
        return [
            'database' => self::validateDatabaseSettings(),
            'mail' => self::validateMailSettings(),
            'cache' => self::validateCacheSettings(),
            'queue' => self::validateQueueSettings(),
            'storage' => self::validateStorageSettings(),
        ];
    }

    /**
     * Check if there are any critical issues
     */
    public static function hasCriticalIssues()
    {
        $validations = self::getAllValidations();
        
        foreach ($validations as $validation) {
            if (!empty($validation['issues'])) {
                return true;
            }
        }
        
        return false;
    }
}