<?php

namespace App\Helpers;

use Illuminate\Support\Facades\File;

class EnvHelper
{
    /**
     * Update a specific key in the .env file
     */
    public static function updateEnvKey($key, $value)
    {
        $envPath = base_path('.env');
        
        \Log::info("Updating .env key: {$key} = {$value}");
        
        if (!File::exists($envPath)) {
            \Log::error(".env file not found at: {$envPath}");
            throw new \Exception('.env file not found');
        }

        $envContent = File::get($envPath);
        
        // Escape special characters in the value
        $escapedValue = str_replace('"', '\\"', $value);
        
        // If the key exists, update it
        if (preg_match("/^{$key}=/m", $envContent)) {
            $envContent = preg_replace("/^{$key}=.*/m", "{$key}=\"{$escapedValue}\"", $envContent);
            \Log::info("Updated existing key: {$key}");
        } else {
            // If the key doesn't exist, add it at the end
            $envContent .= "\n{$key}=\"{$escapedValue}\"";
            \Log::info("Added new key: {$key}");
        }

        try {
            File::put($envPath, $envContent);
            \Log::info("Successfully wrote to .env file");
        } catch (\Exception $e) {
            \Log::error("Failed to write to .env file: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Update multiple keys in the .env file
     */
    public static function updateEnvKeys($settings)
    {
        foreach ($settings as $key => $value) {
            self::updateEnvKey($key, $value);
        }
    }

    /**
     * Get a value from the .env file
     */
    public static function getEnvValue($key, $default = null)
    {
        $envPath = base_path('.env');
        
        if (!File::exists($envPath)) {
            return $default;
        }

        $envContent = File::get($envPath);
        
        if (preg_match("/^{$key}=(.*)/m", $envContent, $matches)) {
            $value = trim($matches[1]);
            // Remove quotes if present
            return trim($value, '"\'');
        }

        return $default;
    }

    /**
     * Check if a key exists in the .env file
     */
    public static function hasEnvKey($key)
    {
        $envPath = base_path('.env');
        
        if (!File::exists($envPath)) {
            return false;
        }

        $envContent = File::get($envPath);
        return preg_match("/^{$key}=/m", $envContent);
    }
}