<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Queue;
use App\Models\User;
use App\Notifications\User\UserRegisteredNotification;

class VerifyQueueConfiguration extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'queue:verify';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Verify queue configuration for notifications';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('🔍 Verifying Queue Configuration...');
        $this->newLine();

        $allChecks = true;

        // 1. Check queue connection
        $allChecks = $this->checkQueueConnection() && $allChecks;
        
        // 2. Check jobs table exists
        $allChecks = $this->checkJobsTable() && $allChecks;
        
        // 3. Check failed_jobs table exists
        $allChecks = $this->checkFailedJobsTable() && $allChecks;
        
        // 4. Check queue status
        $allChecks = $this->checkQueueStatus() && $allChecks;
        
        // 5. Display queue configuration
        $this->displayQueueConfig();
        
        // 6. Test notification dispatch (optional)
        if ($this->confirm('Would you like to test notification dispatch?', false)) {
            $this->testNotificationDispatch();
        }

        $this->newLine();
        
        if ($allChecks) {
            $this->info('✅ All queue configuration checks passed!');
            return Command::SUCCESS;
        } else {
            $this->error('❌ Some queue configuration checks failed. Please review the output above.');
            return Command::FAILURE;
        }
    }

    /**
     * Check queue connection configuration.
     */
    protected function checkQueueConnection(): bool
    {
        $this->info('1. Checking Queue Connection...');
        
        $queueConnection = config('queue.default');
        
        if ($queueConnection === 'sync') {
            $this->error("   ❌ Queue connection is set to 'sync' (synchronous)");
            $this->warn("   📝 Update QUEUE_CONNECTION in .env to 'database'");
            return false;
        }
        
        $this->line("   ✅ Queue connection: <fg=green>{$queueConnection}</>");
        return true;
    }

    /**
     * Check if jobs table exists.
     */
    protected function checkJobsTable(): bool
    {
        $this->info('2. Checking Jobs Table...');
        
        $tableName = config('queue.connections.database.table', 'jobs');
        
        if (!Schema::hasTable($tableName)) {
            $this->error("   ❌ Table '{$tableName}' does not exist");
            $this->warn("   📝 Run: php artisan queue:table && php artisan migrate");
            return false;
        }
        
        $jobCount = DB::table($tableName)->count();
        $this->line("   ✅ Table '{$tableName}' exists (currently {$jobCount} jobs)");
        return true;
    }

    /**
     * Check if failed_jobs table exists.
     */
    protected function checkFailedJobsTable(): bool
    {
        $this->info('3. Checking Failed Jobs Table...');
        
        $tableName = config('queue.failed.table', 'failed_jobs');
        
        if (!Schema::hasTable($tableName)) {
            $this->error("   ❌ Table '{$tableName}' does not exist");
            $this->warn("   📝 Run: php artisan queue:failed-table && php artisan migrate");
            return false;
        }
        
        $failedCount = DB::table($tableName)->count();
        $this->line("   ✅ Table '{$tableName}' exists (currently {$failedCount} failed jobs)");
        
        if ($failedCount > 0) {
            $this->warn("   ⚠️  You have {$failedCount} failed jobs. Consider running: php artisan queue:retry all");
        }
        
        return true;
    }

    /**
     * Check queue status.
     */
    protected function checkQueueStatus(): bool
    {
        $this->info('4. Checking Queue Status...');
        
        try {
            $size = Queue::size();
            $this->line("   ✅ Queue accessible ({$size} pending jobs)");
            
            if ($size > 100) {
                $this->warn("   ⚠️  Queue has {$size} pending jobs. Consider running: php artisan queue:work");
            }
            
            return true;
        } catch (\Exception $e) {
            $this->error("   ❌ Cannot access queue: {$e->getMessage()}");
            return false;
        }
    }

    /**
     * Display queue configuration details.
     */
    protected function displayQueueConfig(): void
    {
        $this->newLine();
        $this->info('📋 Queue Configuration Details:');
        $this->table(
            ['Setting', 'Value'],
            [
                ['Default Connection', config('queue.default')],
                ['Database Driver', config('queue.connections.database.driver')],
                ['Jobs Table', config('queue.connections.database.table', 'jobs')],
                ['Failed Jobs Driver', config('queue.failed.driver')],
                ['Failed Jobs Table', config('queue.failed.table', 'failed_jobs')],
                ['Retry After', config('queue.connections.database.retry_after', 90) . ' seconds'],
            ]
        );

        $this->newLine();
        $this->info('🔔 Notification Queue Priorities:');
        $this->table(
            ['Priority', 'Queue Name', 'Notification Types'],
            [
                ['High', 'high', 'Payment, Verification, OTP, Security'],
                ['Default', 'default', 'Team, Player, Tournament, Match'],
                ['Low', 'low', 'Profile Updates, Password Changes, General Updates'],
            ]
        );
    }

    /**
     * Test notification dispatch.
     */
    protected function testNotificationDispatch(): void
    {
        $this->newLine();
        $this->info('🧪 Testing Notification Dispatch...');
        
        try {
            // Get first user for testing (or create a dummy)
            $user = User::first();
            
            if (!$user) {
                $this->warn('   ⚠️  No users found in database. Skipping test.');
                return;
            }
            
            $beforeCount = Queue::size();
            
            // Dispatch a test notification
            $user->notify(new UserRegisteredNotification($user));
            
            $afterCount = Queue::size();
            
            if ($afterCount > $beforeCount) {
                $this->line("   ✅ Notification dispatched successfully (queue size: {$beforeCount} → {$afterCount})");
                $this->info("   📝 Run 'php artisan queue:work --stop-when-empty' to process it");
            } else {
                $this->warn("   ⚠️  Notification may have been sent synchronously");
                $this->warn("   📝 Check QUEUE_CONNECTION in .env (should be 'database', not 'sync')");
            }
            
        } catch (\Exception $e) {
            $this->error("   ❌ Failed to dispatch test notification: {$e->getMessage()}");
        }
    }
}

