<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\TournamentAlgorithmTester;

class TestTournamentAlgorithms extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'tournament:test-algorithms 
                            {--algorithm= : Test specific algorithm (single_elimination, double_elimination, round_robin, group_knockout, swiss_system)}
                            {--seeding : Test seeding algorithms}
                            {--performance : Run performance tests}
                            {--all : Run all tests}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test tournament bracket algorithms and seeding methods';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $tester = new TournamentAlgorithmTester();

        if ($this->option('all')) {
            $this->runAllTests($tester);
        } elseif ($this->option('algorithm')) {
            $this->runSpecificAlgorithm($tester, $this->option('algorithm'));
        } elseif ($this->option('seeding')) {
            $this->runSeedingTests($tester);
        } elseif ($this->option('performance')) {
            $this->runPerformanceTests($tester);
        } else {
            $this->runInteractiveTests($tester);
        }
    }

    /**
     * Run all tests
     */
    protected function runAllTests(TournamentAlgorithmTester $tester)
    {
        $this->info('Running comprehensive tournament algorithm test suite...');
        
        $results = $tester->runTestSuite();
        
        $this->displayResults($results);
    }

    /**
     * Run specific algorithm test
     */
    protected function runSpecificAlgorithm(TournamentAlgorithmTester $tester, string $algorithm)
    {
        $this->info("Testing {$algorithm} algorithm...");
        
        $method = 'test' . str_replace('_', '', ucwords($algorithm, '_'));
        
        if (method_exists($tester, $method)) {
            $result = $tester->$method();
            $this->displayAlgorithmResult($algorithm, $result);
        } else {
            $this->error("Algorithm '{$algorithm}' not found.");
        }
    }

    /**
     * Run seeding tests
     */
    protected function runSeedingTests(TournamentAlgorithmTester $tester)
    {
        $this->info('Testing seeding algorithms...');
        
        $results = $tester->testSeedingAlgorithms();
        
        $this->displaySeedingResults($results);
    }

    /**
     * Run performance tests
     */
    protected function runPerformanceTests(TournamentAlgorithmTester $tester)
    {
        $this->info('Running performance tests...');
        
        $results = $tester->testPerformance();
        
        $this->displayPerformanceResults($results);
    }

    /**
     * Run interactive tests
     */
    protected function runInteractiveTests(TournamentAlgorithmTester $tester)
    {
        $this->info('Tournament Algorithm Tester');
        $this->line('Choose what to test:');
        
        $choice = $this->choice(
            'Select test type',
            [
                'single_elimination' => 'Single Elimination',
                'double_elimination' => 'Double Elimination',
                'round_robin' => 'Round Robin',
                'group_knockout' => 'Group Knockout',
                'swiss_system' => 'Swiss System',
                'seeding' => 'Seeding Algorithms',
                'advancement' => 'Bracket Advancement',
                'participant_counts' => 'Different Participant Counts',
                'performance' => 'Performance Tests',
                'all' => 'All Tests',
            ]
        );

        switch ($choice) {
            case 'all':
                $this->runAllTests($tester);
                break;
            case 'seeding':
                $this->runSeedingTests($tester);
                break;
            case 'performance':
                $this->runPerformanceTests($tester);
                break;
            case 'advancement':
                $this->info('Testing bracket advancement...');
                $result = $tester->testBracketAdvancement();
                $this->displayAlgorithmResult('advancement', $result);
                break;
            case 'participant_counts':
                $this->info('Testing different participant counts...');
                $results = $tester->testParticipantCounts();
                $this->displayParticipantCountResults($results);
                break;
            default:
                $this->runSpecificAlgorithm($tester, $choice);
        }
    }

    /**
     * Display algorithm test results
     */
    protected function displayAlgorithmResult(string $algorithm, array $result)
    {
        $this->newLine();
        $this->info("✅ {$algorithm} Test Results:");
        
        if ($result['success']) {
            $this->line("Participants: {$result['participants']}");
            $this->line("Total Matches: {$result['total_matches']}");
            
            if (isset($result['rounds'])) {
                $this->line("Rounds: {$result['rounds']}");
            }
            
            if (isset($result['matches_per_round'])) {
                $this->line("Matches per Round: " . implode(', ', $result['matches_per_round']));
            }
            
            if (isset($result['byes'])) {
                $this->line("Byes: {$result['byes']}");
            }
            
            if (isset($result['groups'])) {
                $this->line("Groups: {$result['groups']}");
            }
            
            $this->info("✅ Test passed successfully!");
        } else {
            $this->error("❌ Test failed!");
        }
    }

    /**
     * Display seeding test results
     */
    protected function displaySeedingResults(array $results)
    {
        $this->newLine();
        $this->info("🌱 Seeding Algorithm Test Results:");
        
        foreach ($results as $method => $result) {
            $this->line("{$method}: " . implode(', ', $result['seeding_order']));
        }
        
        $this->info("✅ All seeding tests completed!");
    }

    /**
     * Display performance test results
     */
    protected function displayPerformanceResults(array $results)
    {
        $this->newLine();
        $this->info("⚡ Performance Test Results:");
        
        $table = $this->table(
            ['Participants', 'Execution Time (ms)', 'Total Matches', 'Memory Usage (MB)'],
            collect($results)->map(function ($result, $count) {
                return [
                    $count,
                    $result['execution_time_ms'],
                    $result['total_matches'],
                    $result['memory_usage_mb']
                ];
            })->toArray()
        );
        
        $this->info("✅ Performance tests completed!");
    }

    /**
     * Display participant count test results
     */
    protected function displayParticipantCountResults(array $results)
    {
        $this->newLine();
        $this->info("👥 Participant Count Test Results:");
        
        $table = $this->table(
            ['Participants', 'Total Matches', 'Rounds', 'Byes', 'Power of 2'],
            collect($results)->map(function ($result, $count) {
                return [
                    $count,
                    $result['total_matches'],
                    $result['rounds'],
                    $result['byes'],
                    $result['power_of_two']
                ];
            })->toArray()
        );
        
        $this->info("✅ Participant count tests completed!");
    }

    /**
     * Display comprehensive results
     */
    protected function displayResults(array $results)
    {
        $this->newLine();
        $this->info("🎯 Comprehensive Test Results:");
        
        // Algorithm tests
        $this->line("Algorithm Tests:");
        foreach ($results['algorithms'] as $algorithm => $result) {
            $status = $result['success'] ? '✅' : '❌';
            $this->line("  {$status} {$algorithm}: {$result['participants']} participants, {$result['total_matches']} matches");
        }
        
        // Seeding tests
        $this->line("\nSeeding Tests:");
        foreach ($results['seeding'] as $method => $result) {
            $status = $result['success'] ? '✅' : '❌';
            $this->line("  {$status} {$method}: " . count($result['seeding_order']) . " participants seeded");
        }
        
        // Advancement test
        $this->line("\nAdvancement Test:");
        $advancement = $results['advancement'];
        $status = $advancement['success'] ? '✅' : '❌';
        $this->line("  {$status} Bracket advancement: {$advancement['next_round_matches']} matches created for round {$advancement['next_round']}");
        
        // Performance summary
        $this->line("\nPerformance Summary:");
        foreach ($results['performance'] as $count => $result) {
            $this->line("  {$count} participants: {$result['execution_time_ms']}ms, {$result['memory_usage_mb']}MB");
        }
        
        $this->newLine();
        $this->info("🎉 All tests completed successfully!");
    }
}
