<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class OptimizeSystem extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'system:optimize {--force : Force optimization even in development}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Optimize the entire system for maximum performance';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('🚀 Starting System Optimization...');
        $this->newLine();

        $startTime = microtime(true);

        // Check environment (skip interactive prompt if running from web/without STDIN)
        if (!app()->environment('production') && !$this->option('force')) {
            // Only ask for confirmation if STDIN is available (running from terminal)
            if (defined('STDIN') && stream_isatty(STDIN)) {
                if (!$this->confirm('You are not in production environment. Continue?')) {
                    $this->warn('Optimization cancelled.');
                    return Command::FAILURE;
                }
            } else {
                // Running from web or without terminal, show warning but continue
                $this->warn('Not in production environment, but continuing with optimization...');
            }
        }

        // Step 1: Clear all caches
        $this->optimizeCaches();

        // Step 2: Optimize configurations
        $this->optimizeConfigurations();

        // Step 3: Optimize routes
        $this->optimizeRoutes();

        // Step 4: Optimize views
        $this->optimizeViews();

        // Step 5: Optimize autoloader
        $this->optimizeAutoloader();

        // Step 6: Optimize database
        $this->optimizeDatabase();

        // Step 7: Optimize assets (if in production)
        if (app()->environment('production')) {
            $this->optimizeAssets();
        }

        // Step 8: Warm up caches
        $this->warmUpCaches();

        $executionTime = round(microtime(true) - $startTime, 2);

        $this->newLine();
        $this->info("✅ System optimization completed in {$executionTime} seconds!");
        $this->newLine();

        $this->displayOptimizationStatus();

        return Command::SUCCESS;
    }

    /**
     * Optimize caches.
     */
    protected function optimizeCaches(): void
    {
        $this->info('1️⃣ Optimizing Caches...');
        
        // Clear all caches
        $this->line('   Clearing application cache...');
        Artisan::call('cache:clear');
        
        $this->line('   Clearing config cache...');
        Artisan::call('config:clear');
        
        $this->line('   Clearing route cache...');
        Artisan::call('route:clear');
        
        $this->line('   Clearing view cache...');
        Artisan::call('view:clear');
        
        $this->info('   ✅ Caches cleared');
        $this->newLine();
    }

    /**
     * Optimize configurations.
     */
    protected function optimizeConfigurations(): void
    {
        $this->info('2️⃣ Optimizing Configurations...');
        
        $this->line('   Caching configurations...');
        Artisan::call('config:cache');
        
        $this->info('   ✅ Configurations optimized');
        $this->newLine();
    }

    /**
     * Optimize routes.
     */
    protected function optimizeRoutes(): void
    {
        $this->info('3️⃣ Optimizing Routes...');
        
        $this->line('   Caching routes...');
        Artisan::call('route:cache');
        
        $this->info('   ✅ Routes optimized');
        $this->newLine();
    }

    /**
     * Optimize views.
     */
    protected function optimizeViews(): void
    {
        $this->info('4️⃣ Optimizing Views...');
        
        $this->line('   Compiling views...');
        Artisan::call('view:cache');
        
        $this->info('   ✅ Views optimized');
        $this->newLine();
    }

    /**
     * Optimize autoloader.
     */
    protected function optimizeAutoloader(): void
    {
        $this->info('5️⃣ Optimizing Autoloader...');
        
        $this->line('   Optimizing composer autoloader...');
        
        if (function_exists('exec')) {
            exec('composer dump-autoload --optimize --classmap-authoritative 2>&1', $output, $returnCode);
            
            if ($returnCode === 0) {
                $this->info('   ✅ Autoloader optimized');
            } else {
                $this->warn('   ⚠️  Autoloader optimization skipped (composer not available)');
            }
        } else {
            $this->warn('   ⚠️  Autoloader optimization skipped (exec function disabled)');
        }
        
        $this->newLine();
    }

    /**
     * Optimize database.
     */
    protected function optimizeDatabase(): void
    {
        $this->info('6️⃣ Optimizing Database...');
        
        try {
            // Optimize tables
            $this->line('   Analyzing and optimizing tables...');
            
            $tables = DB::select('SHOW TABLES');
            $database = config('database.connections.' . config('database.default') . '.database');
            $tableKey = 'Tables_in_' . $database;
            
            foreach ($tables as $table) {
                $tableName = $table->$tableKey;
                DB::statement("OPTIMIZE TABLE `{$tableName}`");
                $this->line("   Optimized: {$tableName}");
            }
            
            $this->info('   ✅ Database optimized');
        } catch (\Exception $e) {
            $this->warn('   ⚠️  Database optimization skipped: ' . $e->getMessage());
        }
        
        $this->newLine();
    }

    /**
     * Optimize assets.
     */
    protected function optimizeAssets(): void
    {
        $this->info('7️⃣ Optimizing Assets...');
        
        // Build production assets
        $this->line('   Building production assets...');
        
        if (function_exists('exec')) {
            exec('npm run build 2>&1', $output, $returnCode);
            
            if ($returnCode === 0) {
                $this->info('   ✅ Assets built and optimized');
            } else {
                $this->warn('   ⚠️  Asset optimization skipped (npm not available or build failed)');
            }
        } else {
            $this->warn('   ⚠️  Asset optimization skipped (exec function disabled)');
        }
        
        $this->newLine();
    }

    /**
     * Warm up caches.
     */
    protected function warmUpCaches(): void
    {
        $this->info('8️⃣ Warming Up Caches...');
        
        // Cache frequently accessed data
        $this->line('   Caching frequently accessed data...');
        
        try {
            // Example: Cache settings, configurations, etc.
            // Add your specific cache warming logic here
            
            $this->info('   ✅ Caches warmed up');
        } catch (\Exception $e) {
            $this->warn('   ⚠️  Cache warming skipped: ' . $e->getMessage());
        }
        
        $this->newLine();
    }

    /**
     * Display optimization status.
     */
    protected function displayOptimizationStatus(): void
    {
        $this->info('📊 Optimization Status:');
        $this->newLine();

        $this->table(
            ['Component', 'Status', 'Details'],
            [
                ['Config Cache', $this->checkConfigCached() ? '✅ Cached' : '❌ Not Cached', 'config.php exists'],
                ['Route Cache', $this->checkRouteCached() ? '✅ Cached' : '❌ Not Cached', 'routes.php exists'],
                ['View Cache', $this->checkViewCached() ? '✅ Cached' : '❌ Not Cached', 'Compiled views exist'],
                ['OPCache', $this->checkOpcacheEnabled() ? '✅ Enabled' : '❌ Disabled', 'PHP OPcache status'],
                ['Database', '✅ Optimized', 'Tables analyzed and optimized'],
                ['Autoloader', '✅ Optimized', 'Classmap authoritative'],
            ]
        );

        $this->newLine();
        $this->info('💡 Performance Tips:');
        $this->line('   • Use Redis/Memcached for session and cache');
        $this->line('   • Enable OPcache in PHP configuration');
        $this->line('   • Use CDN for static assets');
        $this->line('   • Enable GZIP compression on web server');
        $this->line('   • Monitor slow queries with Laravel Telescope');
        $this->newLine();
    }

    /**
     * Check if config is cached.
     */
    protected function checkConfigCached(): bool
    {
        return file_exists(base_path('bootstrap/cache/config.php'));
    }

    /**
     * Check if routes are cached.
     */
    protected function checkRouteCached(): bool
    {
        return file_exists(base_path('bootstrap/cache/routes-v7.php'));
    }

    /**
     * Check if views are cached.
     */
    protected function checkViewCached(): bool
    {
        $viewPath = storage_path('framework/views');
        return is_dir($viewPath) && count(scandir($viewPath)) > 2;
    }

    /**
     * Check if OPcache is enabled.
     */
    protected function checkOpcacheEnabled(): bool
    {
        return function_exists('opcache_get_status') && opcache_get_status() !== false;
    }
}

