<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\NotificationSetting;
use App\Services\NotificationRegistryService;

class ListNotifications extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'notifications:list 
                            {--category= : Filter by category}
                            {--disabled : Show only disabled notifications}
                            {--enabled : Show only enabled notifications}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'List all notifications with their status';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $category = $this->option('category');
        $showDisabled = $this->option('disabled');
        $showEnabled = $this->option('enabled');

        // Build query
        $query = NotificationSetting::query();

        if ($category) {
            $query->where('category', $category);
        }

        if ($showDisabled) {
            $query->where('enabled', false);
        }

        if ($showEnabled) {
            $query->where('enabled', true);
        }

        $notifications = $query->orderBy('category')->orderBy('notification_type')->get();

        if ($notifications->isEmpty()) {
            $this->warn('No notifications found matching the criteria.');
            $this->info('💡 Run "php artisan notifications:sync" to sync notifications from code.');
            return Command::SUCCESS;
        }

        // Group by category
        $grouped = $notifications->groupBy('category');

        $this->info('📧 Notification Settings');
        $this->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');

        foreach ($grouped as $cat => $items) {
            $this->info("\n📁 " . strtoupper($cat) . " ({$items->count()} notifications)");
            $this->info(str_repeat('─', 60));

            $tableData = [];
            foreach ($items as $notification) {
                $status = $notification->enabled ? '✅ Enabled' : '❌ Disabled';
                $channels = implode(', ', $notification->channels ?? ['mail', 'database']);
                
                $tableData[] = [
                    $notification->notification_type,
                    $status,
                    $channels,
                    $notification->description ?? 'N/A',
                ];
            }

            $this->table(
                ['Notification Type', 'Status', 'Channels', 'Description'],
                $tableData
            );
        }

        // Show summary
        $totalEnabled = $notifications->where('enabled', true)->count();
        $totalDisabled = $notifications->where('enabled', false)->count();
        
        $this->info("\n📊 Summary:");
        $this->info("  Total: {$notifications->count()}");
        $this->info("  ✅ Enabled: {$totalEnabled}");
        $this->info("  ❌ Disabled: {$totalDisabled}");

        return Command::SUCCESS;
    }
}

