# Category-Based Location Management Implementation

**Date**: October 18, 2025  
**Status**: ✅ Complete

## Overview

Implemented a flexible category-based location management system that allows administrators to configure location restrictions per tournament category (Men, Women, Youth Male, Youth Female, Children).

## What Changed

### Backend Changes

#### 1. Configuration (`config/tournament.php`)
- Added global `available_countries` and `available_counties` settings
- Added `category_locations` structure for each category:
  ```php
  'category_locations' => [
      'men' => [
          'restrict_locations' => false,  // Toggle ON/OFF
          'locations' => [],              // Allowed locations when restricted
      ],
      // ... other categories
  ]
  ```

#### 2. Tournament Settings Controller (`app/Http/Controllers/Admin/TournamentSettingsController.php`)
- Added validation for `category_locations` structure
- Implemented `updateConfigFile()` method to write category locations to config
- Added automatic cache clearing after config file updates:
  - `config:clear` - Clears cached config
  - `config:cache` - Rebuilds config cache
  - `opcache_reset()` - Clears PHP OPcache
- Changed JSON response to Inertia redirect for proper response handling

#### 3. Registration Controller (`app/Http/Controllers/Auth/RegisteredUserController.php`)
- Passes `category_locations` data to frontend
- Passes global `countries` and `counties` arrays
- Ensures `default_tournament` includes `available_categories`

### Frontend Changes

#### 1. Tournament Settings UI (`resources/js/pages/admin/tournament-settings/Index.tsx`)
- Added "Global Location Settings" card:
  - Available Countries (TagsInput)
  - Available Counties (TagsInput)
- Added "Category Location Settings" card:
  - Each category displayed with icon and description
  - Switch toggle for "Restrict Locations"
  - TagsInput for locations (disabled when toggle OFF, editable when ON)
  - Visual feedback showing restriction status

#### 2. Team Registration (`resources/js/pages/frontend/auth/RegisterTeam.tsx`)
- **Category Selection**: Filters to show only enabled categories for selected tournament
- **Dynamic Location Input**:
  - If category has restrictions: Dropdown with allowed locations
  - If no restrictions: Free text input for any location
- **Player Location**: Same restriction logic applied to player locations
- Helper text indicates restriction status

#### 3. UI Components

**CategorySelector (`resources/js/components/CategorySelector.tsx`):**
- Fixed category re-selection issue
- Proper label wrapping for radio inputs
- Categories can be navigated back and forth

**TagsInput (`resources/js/components/ui/tags-input.tsx`):**
- Added `disabled` prop support
- Visual styling for disabled state (gray background)
- Remove buttons hidden when disabled

**Tournament Admin Forms:**
- Converted all boolean toggles from Checkboxes to Switches:
  - Auto-accept Registrations
  - Allow Spectators
  - Require Approval
  - Featured Tournament
  - Enable Streaming
- Category enable/disable uses Switch (not checkbox)
- Fee inputs hidden when category is disabled

## Files Modified

### Configuration
- `config/tournament.php`

### Backend
- `app/Http/Controllers/Admin/TournamentSettingsController.php`
- `app/Http/Controllers/Auth/RegisteredUserController.php`

### Frontend - Admin
- `resources/js/pages/admin/tournament-settings/Index.tsx`
- `resources/js/pages/admin/tournaments/edit.tsx`
- `resources/js/components/tournaments/form-steps/tournament-setup-step.tsx`
- `resources/js/components/tournaments/form-steps/schedule-participants-step.tsx`
- `resources/js/components/tournaments/form-steps/media-settings-step.tsx`

### Frontend - Team Registration
- `resources/js/pages/frontend/auth/RegisterTeam.tsx`
- `resources/js/components/CategorySelector.tsx`
- `resources/js/components/ui/tags-input.tsx`

### Documentation
- `docs/documentation/08_Core_Features.md`
- `docs/documentation/17_User_Guide.md`
- `docs/02_Tournament_Management_Agenda.md`
- `docs/documentation/18_Developer_Guide.md`

## How It Works

### Admin Configuration Flow

1. Navigate to **Admin → Tournament Settings**
2. Configure global countries and counties (applies to unrestricted categories)
3. For each category:
   - Toggle "Restrict Locations" switch
   - If ON: Add allowed locations using tags input
   - If OFF: Location input is disabled (no restrictions)
4. Click "Save Settings"
5. Config cache automatically clears and rebuilds

### Team Registration Flow

1. User selects tournament
2. User selects category (only sees enabled categories)
3. Location field adapts:
   - **Restricted**: Dropdown showing allowed locations with helper text
   - **Unrestricted**: Text input for any location with helper text
4. Player locations follow same restriction logic
5. Submit registration with category-validated location

## Benefits

✅ **Flexible Configuration**: Categories can be restricted or unrestricted independently  
✅ **Better UX**: Clear visual indicators (switches, conditional inputs)  
✅ **Automatic Updates**: Config cache clears automatically on save  
✅ **Consistent UI**: All boolean toggles now use switches  
✅ **Category Filtering**: Only enabled categories shown to users  
✅ **Dynamic Forms**: Location inputs adapt based on restrictions  

## Technical Details

### Database Schema
- Teams table has `country`, `county`, `location` fields
- Tournaments table has `enable_categories`, `category_fees`, `available_categories` fields

### Configuration Storage
- Global settings: Stored in `.env` file
- Category locations: Stored directly in `config/tournament.php`
- Both automatically cache-cleared on updates

### Frontend State Management
- Inertia.js for server-side rendering
- React state for form management
- Dynamic rendering based on category selection

## Testing Checklist

- [x] Admin can configure global locations
- [x] Admin can toggle category restrictions
- [x] Admin can add locations to restricted categories
- [x] TagsInput disabled when toggle is OFF
- [x] Config cache clears automatically on save
- [x] Team registration shows only enabled categories
- [x] Location input shows dropdown for restricted categories
- [x] Location input shows text field for unrestricted categories
- [x] Player locations follow same restriction logic
- [x] Category re-selection works properly
- [x] All switches work correctly in tournament forms

## Future Enhancements

- County-based location filtering (County → Location cascading)
- Tournament-specific location overrides
- Location import from external sources
- Location geocoding and validation

