<!-- 84cd2fd1-c4af-40fa-9d35-67980159af2d a28ad777-4272-4add-8f70-72e635bb7b5f -->
# Tournament Category System - Complete Implementation

## Overview

Implement a comprehensive tournament category system allowing tournaments to have multiple categories (Men, Women, Youth Male, Youth Female, Children) with different registration fees for each category. This affects the entire team registration and management flow.

## Database Changes

### 1. Create Migration: Add Category to Teams Table

**File:** `database/migrations/YYYY_MM_DD_add_category_to_teams_table.php`

**Changes:**

- Add `category` enum field: men, women, youth_male, youth_female, children
- Add index on category field
- Set default to 'men'

### 2. Create Migration: Add Category Fees to Tournaments Table

**File:** `database/migrations/YYYY_MM_DD_add_category_fees_to_tournaments_table.php`

**Changes:**

- Add `category_fees` JSON field to store fees per category
- Add `enable_categories` boolean field (default false for backward compatibility)
- Add `available_categories` JSON field (which categories are allowed)

### 3. Update Team Model

**File:** `app/Models/Team.php`

**Add:**

- `category` to fillable
- Cast category as string
- Validation method for category eligibility
- Helper methods: `isMensTeam()`, `isWomensTeam()`, `isYouthTeam()`, `isChildrensTeam()`

### 4. Update Tournament Model

**File:** `app/Models/Tournament.php`

**Add:**

- `category_fees` to fillable and cast as array
- `enable_categories` to fillable and cast as boolean
- `available_categories` to fillable and cast as array
- Method `getFeeForCategory($category)` - returns fee for specific category
- Method `isCategoryAllowed($category)` - checks if category is enabled
- Default category_fees structure

## Backend Logic Updates

### 5. Update Team Registration Controller

**File:** `app/Http/Controllers/Auth/RegisteredUserController.php`

**Changes:**

- Add `team.category` to validation rules
- Validate category is allowed for tournament
- Use category-specific fee for payment creation
- Validate player ages match category requirements
- Validate player genders match category requirements (where applicable)

### 6. Update Team Controller

**File:** `app/Http/Controllers/TeamController.php`

**Changes:**

- Add category to team update validation
- Prevent category change after approval
- Add category filter methods

### 7. Update Team Admin Controller

**File:** `app/Http/Controllers/Admin/TeamAdminController.php`

**Changes:**

- Add category field to team creation/editing
- Add category filters to team lists
- Show category-specific fees in admin views

### 8. Create Category Validation Service

**File:** `app/Services/CategoryValidationService.php`

**Methods:**

- `validatePlayerAgeForCategory($playerAge, $category)` - Check age requirements
- `validatePlayerGenderForCategory($playerGender, $category)` - Check gender requirements
- `getCategoryRequirements($category)` - Get age/gender requirements
- `getCategoryFeeForTournament($tournament, $category)` - Get specific fee

## Frontend Updates

### 9. Update Registration Form

**File:** `resources/js/pages/frontend/auth/Register.tsx` or `RegisterTeamNew.tsx`

**Changes:**

- Add category selection dropdown (Step 2)
- Show category requirements (age, gender)
- Display category-specific fee dynamically
- Add category validation
- Update payment calculation
- Show category badge in form

### 10. Update Team Dashboard

**File:** `resources/js/components/dashboard/TeamManagerDashboard.tsx`

**Changes:**

- Display team category badge
- Show category-specific information
- Filter/group by category if needed

### 11. Update Team Registration Alert

**File:** `resources/js/components/TeamManager/TeamRegistrationAlert.tsx`

**Changes:**

- Show category in alert
- Display category-specific fee amounts
- Category-aware messaging

### 12. Create Category Badge Component

**File:** `resources/js/components/ui/category-badge.tsx`

**Features:**

- Color-coded badges per category
- Icons for each category
- Responsive design

## Config Updates

### 13. Update Tournament Config

**File:** `config/tournament.php`

**Add:**

```php
'categories' => [
    'men' => [
        'name' => 'Men',
        'age_min' => 18,
        'age_max' => null,
        'gender' => 'male',
        'default_fee' => 1000,
        'icon' => '👨',
        'color' => 'blue'
    ],
    'women' => [
        'name' => 'Women',
        'age_min' => 18,
        'age_max' => null,
        'gender' => 'female',
        'default_fee' => 800,
        'icon' => '👩',
        'color' => 'pink'
    ],
    // ... etc for all 5 categories
]
```

## Documentation Updates

### 14. Update Team Registration Documentation

**File:** `resources/views/documentation/partials/team-registration.blade.php`

**Add:**

- New step for category selection
- Category requirements table
- Category fee structure
- Updated flow diagram showing category selection
- Category validation decision points

### 15. Create Tournament Categories Tab

**File:** `resources/views/documentation/partials/tournament-categories.blade.php`

**Content:**

- Category system overview
- 5 categories with requirements
- Fee structure comparison table
- Eligibility rules
- Category selection flow diagram
- Mixed team handling
- Category change restrictions

### 16. Add Tab to Main Documentation

**File:** `resources/views/documentation/system-walkthrough.blade.php`

**Add:**

- New tab button "🎯 Tournament Categories"
- Tab content div with include
- Position after Tournament Management

## Validation & Business Rules

### 17. Category Eligibility Rules

**Men's Teams:**

- All players must be 18+ years old
- All players must be male
- No age upper limit

**Women's Teams:**

- All players must be 18+ years old
- All players must be female
- No age upper limit

**Youth Male Teams:**

- All players must be 13-17 years old
- All players must be male
- Strict age enforcement

**Youth Female Teams:**

- All players must be 13-17 years old
- All players must be female
- Strict age enforcement

**Children's Teams:**

- All players must be under 13 years old
- Any gender allowed
- Mixed teams permitted

### 18. Category Fee Logic

**Fee Determination:**

1. Check if tournament has `category_fees` configured
2. If yes, use tournament-specific fee for category
3. If no, use default fee from config
4. If category not available, show error

**Fee Display:**

- Show all available category fees during tournament selection
- Highlight selected category fee
- Update payment amount dynamically

## Migration Strategy (Safe Implementation)

### Phase 1: Database (Non-Breaking)

1. Add new fields with defaults
2. Make fields nullable initially
3. Run migration
4. Verify no errors

### Phase 2: Backend (Backward Compatible)

1. Update models with new fields
2. Make category optional in validation
3. Add helper methods
4. Default to 'men' if not specified

### Phase 3: Frontend (Progressive Enhancement)

1. Add category selection UI
2. Make it optional initially
3. Show only if tournament has categories enabled
4. Graceful fallback to single fee

### Phase 4: Validation (Gradual Enforcement)

1. Add soft validation (warnings)
2. Log validation issues
3. Enable hard validation per tournament
4. Full enforcement after testing

### Phase 5: Documentation

1. Update existing docs
2. Add new category section
3. Update diagrams
4. Add migration guide

## Testing Strategy

### Unit Tests

- Category validation service tests
- Fee calculation tests
- Age requirement tests
- Gender requirement tests

### Integration Tests

- Complete registration with categories
- Payment amount verification
- Category change restrictions
- Mixed team validation

### UI Tests

- Category selection appears correctly
- Fees update dynamically
- Validation messages show
- Category badges display

## Rollout Plan

### Step 1: Enable for New Tournaments Only

- Existing tournaments unaffected
- New tournaments can opt-in to categories
- Test with limited audience

### Step 2: Migrate Existing Teams

- Categorize existing teams based on player data
- Update payment records if needed
- Notify teams of category assignment

### Step 3: Full Rollout

- Enable for all new registrations
- Update all documentation
- Train support team

## File Checklist

### Backend (8 files)

- [ ] Migration: add_category_to_teams_table.php
- [ ] Migration: add_category_fees_to_tournaments_table.php
- [ ] Model: Team.php (update)
- [ ] Model: Tournament.php (update)
- [ ] Service: CategoryValidationService.php (new)
- [ ] Controller: RegisteredUserController.php (update)
- [ ] Controller: TeamController.php (update)
- [ ] Controller: TeamAdminController.php (update)

### Frontend (5 files)

- [ ] Register.tsx or RegisterTeamNew.tsx (update)
- [ ] TeamManagerDashboard.tsx (update)
- [ ] TeamRegistrationAlert.tsx (update)
- [ ] category-badge.tsx (new component)
- [ ] CategorySelector.tsx (new component)

### Config (1 file)

- [ ] config/tournament.php (update)

### Documentation (3 files)

- [ ] partials/team-registration.blade.php (update)
- [ ] partials/tournament-categories.blade.php (new)
- [ ] system-walkthrough.blade.php (add tab)

**Total Files:** 17

## Benefits

✅ **Fair Pricing** - Different fees for different categories

✅ **Inclusive** - Supports women, youth, and children

✅ **Flexible** - Tournament organizers control categories

✅ **Validated** - Age and gender requirements enforced

✅ **Backward Compatible** - Existing tournaments unaffected

✅ **Well Documented** - Complete workflow documentation

This comprehensive implementation ensures all aspects of the system are updated cohesively while maintaining backward compatibility.